/**
 * Converge Admin JS — Settings handlers, company form, job editing, competency management,
 * candidate review, analytics charts, and all interactive features.
 */
(function ($) {
  "use strict";

  // --- Utility: Relative Time ---

  function relativeTime(dateStr) {
    if (!dateStr) return "";
    var now = new Date();
    var date = new Date(dateStr);
    var diff = Math.floor((now - date) / 1000);
    if (diff < 60) return "just now";
    if (diff < 3600) return Math.floor(diff / 60) + "m ago";
    if (diff < 86400) return Math.floor(diff / 3600) + "h ago";
    if (diff < 604800) return Math.floor(diff / 86400) + "d ago";
    return date.toLocaleDateString();
  }

  // Update all relative timestamps on page load
  $(".converge-candidate-time[data-timestamp]").each(function () {
    var ts = $(this).data("timestamp");
    if (ts) $(this).text(relativeTime(ts));
  });

  // --- Utility: Copy to Clipboard ---

  $(document).on("click", ".converge-copy-btn", function () {
    var $btn = $(this);
    var text = $btn.data("copy");
    if (!text) return;

    navigator.clipboard.writeText(text).then(function () {
      var orig = $btn.text();
      $btn.text("Copied!");
      setTimeout(function () { $btn.text(orig); }, 2000);
    }).catch(function () {
      // Fallback
      var $temp = $('<textarea>').val(text).appendTo('body');
      $temp[0].select();
      document.execCommand('copy');
      $temp.remove();
      var orig = $btn.text();
      $btn.text("Copied!");
      setTimeout(function () { $btn.text(orig); }, 2000);
    });
  });

  // --- Company Tab Switching ---

  $(document).on("click", ".converge-company-tab", function () {
    var tab = $(this).data("tab");
    $(".converge-company-tab").removeClass("is-active");
    $(this).addClass("is-active");
    $(".converge-tab-panel").removeClass("is-active");
    $('[data-tab-panel="' + tab + '"]').addClass("is-active");
  });

  // --- Office Locations CRUD ---

  function collectLocations() {
    var locations = [];
    $(".converge-location-item").each(function () {
      var name = $(this).find(".loc-name").val() || "";
      var address = $(this).find(".loc-address").val() || "";
      var isHq = $(this).find(".loc-hq").is(":checked");
      if (name || address) {
        locations.push({ name: name, address: address, is_hq: isHq });
      }
    });
    return locations;
  }

  $("#converge-add-location").on("click", function () {
    var html =
      '<div class="converge-location-item">' +
        '<div class="converge-location-fields">' +
          '<input type="text" class="loc-name" value="" placeholder="Name (e.g. HQ)">' +
          '<input type="text" class="loc-address" value="" placeholder="Address" style="flex: 2;">' +
          '<label><input type="checkbox" class="loc-hq" value="1"> HQ</label>' +
        '</div>' +
        '<button type="button" class="converge-location-remove" title="Remove">&times;</button>' +
      '</div>';
    $("#converge-locations-list").append(html);
  });

  $(document).on("click", ".converge-location-remove", function () {
    $(this).closest(".converge-location-item").remove();
  });

  // --- Company Form AJAX Save ---

  $("#converge-company-form").on("submit", function (e) {
    e.preventDefault();

    var $form = $(this);
    var $btn = $form.find('button[type="submit"]');
    var $status = $("#converge-company-save-status");

    $btn.prop("disabled", true);
    $status.text("Saving...").removeClass("converge-save-error").addClass("converge-save-active");

    // Serialize office locations before sending
    $("#converge-locations-data").val(JSON.stringify(collectLocations()));

    var data = {
      action: "converge_save_company_info",
      nonce: convergeAdmin.companyNonce,
    };

    // Collect all form fields across all tabs
    $form.find("input, select, textarea").each(function () {
      var name = $(this).attr("name");
      if (!name || name === "office_locations") return;

      if ($(this).is(":checkbox")) {
        data[name] = $(this).is(":checked") ? "1" : "";
      } else {
        data[name] = $(this).val();
      }
    });

    // Add serialized locations
    data["office_locations"] = $("#converge-locations-data").val();

    $.post(convergeAdmin.ajaxUrl, data)
      .done(function (response) {
        if (response.success) {
          $status.text("Saved!").removeClass("converge-save-active");
          setTimeout(function () { $status.text(""); }, 3000);
        } else {
          $status.text("Error: " + (response.data || "Save failed")).addClass("converge-save-error").removeClass("converge-save-active");
        }
      })
      .fail(function () {
        $status.text("Request failed. Please try again.").addClass("converge-save-error").removeClass("converge-save-active");
      })
      .always(function () {
        $btn.prop("disabled", false);
      });
  });

  // --- Job Form AJAX Save ---

  $(".converge-job-form").on("submit", function (e) {
    e.preventDefault();

    var $form = $(this);
    var $btn = $form.find('button[type="submit"]');
    var $status = $form.find(".converge-save-status").not(".converge-gen-status");
    var section = $form.data("section");
    var jobId = $form.data("job-id");

    $btn.prop("disabled", true);
    $status.text("Saving...").removeClass("converge-save-error").addClass("converge-save-active");

    var data = {
      action: "converge_update_job",
      nonce: convergeAdmin.jobNonce,
      job_id: jobId,
      section: section,
    };

    // Special handling for competencies section
    if (section === "competencies") {
      data.competencies = JSON.stringify(collectCompetencies());
    } else if (section === "conversation") {
      data.conversation_config = JSON.stringify(collectConversationConfig());
    } else {
      // Collect fields based on section
      $form.find("input, select, textarea").each(function () {
        var name = $(this).attr("name");
        if (!name || name === "competencies") return;

        if ($(this).is(":checkbox")) {
          if (name === "allowed_languages[]") {
            if (!data["allowed_languages[]"]) {
              data["allowed_languages[]"] = [];
            }
            if ($(this).is(":checked")) {
              data["allowed_languages[]"].push($(this).val());
            }
          } else {
            data[name] = $(this).is(":checked") ? "1" : "";
          }
        } else {
          data[name] = $(this).val();
        }
      });

      // For AI section, sync voice_name with voice selection
      if (section === "ai") {
        var voiceVal = $form.find("#ai-voice").val();
        data["voice_name"] = voiceVal;
        data["voice"] = voiceVal;
      }

      // For screening section, collect custom requirements
      if (section === "screening") {
        data.custom_requirements = JSON.stringify(collectCustomRequirements());
      }
    }

    $.post(convergeAdmin.ajaxUrl, data)
      .done(function (response) {
        if (response.success) {
          $status.text("Saved!").removeClass("converge-save-active");
          setTimeout(function () { $status.text(""); }, 3000);
        } else {
          $status
            .text("Error: " + (response.data || "Save failed"))
            .addClass("converge-save-error")
            .removeClass("converge-save-active");
        }
      })
      .fail(function () {
        $status
          .text("Request failed. Please try again.")
          .addClass("converge-save-error")
          .removeClass("converge-save-active");
      })
      .always(function () {
        $btn.prop("disabled", false);
      });
  });

  // --- Collect Conversation Config ---

  function collectConversationConfig() {
    return {
      background: {
        enabled: $('input[name="bg_enabled"]').is(":checked"),
        depth: parseInt($('input[name="bg_depth"]').val()) || 2,
        focus_areas: {
          career_trajectory: $('input[name="bg_career_trajectory"]').is(":checked"),
          recent_role: $('input[name="bg_recent_role"]').is(":checked"),
          key_achievements: $('input[name="bg_key_achievements"]').is(":checked"),
          role_transitions: $('input[name="bg_role_transitions"]').is(":checked"),
        },
      },
      motivation: {
        enabled: $('input[name="mo_enabled"]').is(":checked"),
        questions: {
          why_this_role: $('input[name="mo_why_this_role"]').is(":checked"),
          why_this_company: $('input[name="mo_why_this_company"]').is(":checked"),
          why_looking_now: $('input[name="mo_why_looking_now"]').is(":checked"),
          career_goals: $('input[name="mo_career_goals"]').is(":checked"),
          what_matters_most: $('input[name="mo_what_matters_most"]').is(":checked"),
        },
      },
      culture_fit: {
        enabled: $('input[name="cf_enabled"]').is(":checked"),
        areas: {
          work_environment: $('input[name="cf_work_environment"]').is(":checked"),
          management_style: $('input[name="cf_management_style"]').is(":checked"),
          collaboration_style: $('input[name="cf_collaboration_style"]').is(":checked"),
          work_life_balance: $('input[name="cf_work_life_balance"]').is(":checked"),
          values_alignment: $('input[name="cf_values_alignment"]').is(":checked"),
        },
        probe_company_values: $('input[name="cf_probe_values"]').is(":checked"),
      },
    };
  }

  // --- Collect Custom Requirements ---

  function collectCustomRequirements() {
    var reqs = [];
    $(".converge-custom-req-item").each(function () {
      var question = $(this).find(".custom-req-question").val() || "";
      var disqualifyOn = $(this).find(".custom-req-disqualify").val() || "no";
      if (question.trim()) {
        reqs.push({ question: question.trim(), disqualify_on: disqualifyOn });
      }
    });
    return reqs;
  }

  // Add custom requirement
  $(document).on("click", "#converge-add-custom-req", function () {
    var html =
      '<div class="converge-custom-req-item">' +
        '<input type="text" class="custom-req-question regular-text" value="" placeholder="e.g., Do you have CPR certification?">' +
        '<select class="custom-req-disqualify">' +
          '<option value="no">Disqualify on No</option>' +
          '<option value="yes">Disqualify on Yes</option>' +
        '</select>' +
        '<button type="button" class="converge-custom-req-remove" title="Remove">&times;</button>' +
      '</div>';
    $("#converge-custom-requirements").append(html);
  });

  $(document).on("click", ".converge-custom-req-remove", function () {
    $(this).closest(".converge-custom-req-item").remove();
  });

  // --- Depth Slider Display (all depth sliders) ---

  $(document).on("input", ".converge-depth-slider", function () {
    $(this).closest(".converge-range-wrap").find(".converge-range-value").text($(this).val());
  });

  // --- Competency Card Expand/Collapse ---

  $(document).on("click", ".converge-competency-header", function (e) {
    if ($(e.target).closest(".converge-competency-delete").length) return;
    var $card = $(this).closest(".converge-competency-card");
    $card.toggleClass("is-expanded");
  });

  // --- Competency Name Sync ---

  $(document).on("input", ".comp-name", function () {
    var $card = $(this).closest(".converge-competency-card");
    var name = $(this).val() || "New Competency";
    $card.find(".converge-competency-name").text(name);
  });

  // --- Competency Weight Sync ---

  $(document).on("input", ".comp-weight", function () {
    var $card = $(this).closest(".converge-competency-card");
    var weight = $(this).val() || "0";
    $card.find(".converge-competency-weight").text(weight + "%");
  });

  // --- Add Competency ---

  $("#converge-add-competency").on("click", function () {
    var idx = $(".converge-competency-card").length;
    var html =
      '<div class="converge-competency-card is-expanded" data-index="' + idx + '">' +
        '<div class="converge-competency-header">' +
          '<span class="dashicons dashicons-arrow-right-alt2 converge-competency-expand"></span>' +
          '<span class="converge-competency-name">New Competency</span>' +
          '<span class="converge-competency-weight">0%</span>' +
          '<button type="button" class="converge-competency-delete" title="Delete">&times;</button>' +
        '</div>' +
        '<div class="converge-competency-body">' +
          '<label>Name</label>' +
          '<input type="text" class="comp-name regular-text" value="">' +
          '<label>Weight (%)</label>' +
          '<input type="number" class="comp-weight" value="0" min="0" max="100" style="width: 80px;">' +
          '<label>Signals (one per line)</label>' +
          '<textarea class="comp-signals" rows="3"></textarea>' +
          '<label>Sample Questions (one per line)</label>' +
          '<textarea class="comp-questions" rows="3"></textarea>' +
          '<div class="converge-competency-actions">' +
            '<button type="button" class="button button-small converge-generate-questions">Generate Questions</button>' +
            '<span class="converge-save-status converge-gen-status"></span>' +
          '</div>' +
        '</div>' +
      '</div>';

    $("#converge-competencies-list").append(html);

    var $new = $("#converge-competencies-list .converge-competency-card:last");
    $new[0].scrollIntoView({ behavior: "smooth", block: "center" });
    $new.find(".comp-name").focus();
  });

  // --- Delete Competency ---

  $(document).on("click", ".converge-competency-delete", function () {
    if (!confirm("Delete this competency?")) return;
    $(this).closest(".converge-competency-card").remove();
  });

  // --- Collect Competencies into JSON ---

  function collectCompetencies() {
    var competencies = [];
    $(".converge-competency-card").each(function () {
      var $card = $(this);
      var name = $card.find(".comp-name").val() || "";
      var weight = parseFloat($card.find(".comp-weight").val() || "0") / 100;
      var signals = ($card.find(".comp-signals").val() || "")
        .split("\n")
        .map(function (s) { return s.trim(); })
        .filter(function (s) { return s.length > 0; });
      var questions = ($card.find(".comp-questions").val() || "")
        .split("\n")
        .map(function (s) { return s.trim(); })
        .filter(function (s) { return s.length > 0; });

      if (name) {
        competencies.push({
          name: name,
          weight: weight,
          signals: signals,
          sampleQuestions: questions,
        });
      }
    });
    return competencies;
  }

  // --- Generate Questions per Competency ---

  $(document).on("click", ".converge-generate-questions", function () {
    var $btn = $(this);
    var $card = $btn.closest(".converge-competency-card");
    var $status = $card.find(".converge-gen-status");
    var compName = $card.find(".comp-name").val();
    var signals = $card.find(".comp-signals").val();

    if (!compName) {
      $status.text("Enter a competency name first.").addClass("converge-save-error");
      return;
    }

    var jobTitle = $("#job-title").val() || "";
    var language = $('input[name="language"]').val() || "en";

    $btn.prop("disabled", true);
    $status
      .text("Generating...")
      .removeClass("converge-save-error")
      .addClass("converge-save-active");

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_generate_questions",
      nonce: convergeAdmin.jobNonce,
      competency_name: compName,
      signals: signals,
      job_title: jobTitle,
      language: language,
    })
      .done(function (response) {
        if (response.success && response.data && response.data.questions) {
          var questions = response.data.questions;
          if (Array.isArray(questions)) {
            $card.find(".comp-questions").val(questions.join("\n"));
            $status.text("Generated!").removeClass("converge-save-active");
          } else {
            $status
              .text("Unexpected response format")
              .addClass("converge-save-error")
              .removeClass("converge-save-active");
          }
        } else {
          $status
            .text("Error: " + (response.data || "Generation failed"))
            .addClass("converge-save-error")
            .removeClass("converge-save-active");
        }
        setTimeout(function () { $status.text(""); }, 3000);
      })
      .fail(function () {
        $status
          .text("Request failed.")
          .addClass("converge-save-error")
          .removeClass("converge-save-active");
      })
      .always(function () {
        $btn.prop("disabled", false);
      });
  });

  // --- Parse Competencies Button ---

  $(document).on("click", "#converge-parse-competencies", function () {
    var $btn = $(this);
    var $status = $("#converge-parse-status");
    var jobId = $btn.data("job-id");
    var description = $btn.data("description");

    if (!description) {
      $status
        .text("No job description available to parse.")
        .addClass("converge-save-error");
      return;
    }

    $btn.prop("disabled", true);
    $status
      .text("Parsing competencies... this may take 15-30 seconds.")
      .removeClass("converge-save-error")
      .addClass("converge-save-active");

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_parse_competencies",
      nonce: convergeAdmin.jobNonce,
      job_id: jobId,
      description: description,
    })
      .done(function (response) {
        if (response.success) {
          $status.text("Competencies parsed! Reloading...").removeClass("converge-save-active");
          setTimeout(function () { location.reload(); }, 1000);
        } else {
          $status
            .text("Error: " + (response.data || "Parse failed"))
            .addClass("converge-save-error")
            .removeClass("converge-save-active");
          $btn.prop("disabled", false);
        }
      })
      .fail(function () {
        $status
          .text("Request failed. Please try again.")
          .addClass("converge-save-error")
          .removeClass("converge-save-active");
        $btn.prop("disabled", false);
      });
  });

  // --- Screening Requirement Toggles (show/hide behavior dropdown) ---

  $(document).on("change", ".converge-requirement-toggle", function () {
    var $row = $(this).closest(".converge-toggle-row");
    var $behavior = $row.find(".converge-requirement-behavior");
    if ($(this).is(":checked")) {
      $behavior.addClass("is-visible");
    } else {
      $behavior.removeClass("is-visible");
    }
  });

  // --- Probing Depth Range Display ---

  $("#ai-probing-depth").on("input", function () {
    $("#ai-depth-display").text($(this).val());
  });

  // --- Voice Selection + Preview ---

  var voiceAudio = null;
  var voicePlaying = false;

  function updateVoiceDetail() {
    var $sel = $("#ai-voice");
    var voicesData = $sel.data("voices") || {};
    var key = $sel.val();
    var v = voicesData[key];

    $("#ai-voice-name").val(key);

    if (v) {
      $("#voice-detail-name").text(v.label);
      $("#voice-detail-gender").text(v.gender);
      $("#voice-detail-desc").text(v.desc);
      if (v.recommended) {
        $("#voice-detail-rec").show();
      } else {
        $("#voice-detail-rec").hide();
      }
    }
  }

  function stopVoicePreview() {
    if (voiceAudio) {
      voiceAudio.pause();
      voiceAudio.currentTime = 0;
      voiceAudio = null;
    }
    voicePlaying = false;
    var $btn = $("#converge-voice-preview");
    $btn.removeClass("is-playing").prop("disabled", false);
    $btn.find(".dashicons")
      .removeClass("dashicons-controls-pause")
      .addClass("dashicons-controls-play");
    $("#voice-playing-text").hide();
  }

  $("#ai-voice").on("change", function () {
    stopVoicePreview();
    updateVoiceDetail();
  });

  $("#converge-voice-preview").on("click", function () {
    var $btn = $(this);

    if (voicePlaying) {
      stopVoicePreview();
      return;
    }

    var $sel = $("#ai-voice");
    var $opt = $sel.find("option:selected");
    var voiceId = $opt.data("voice-id");
    var voiceName = $opt.data("hume-name");

    if (!voiceId) return;

    $btn.prop("disabled", true);
    $btn.find(".dashicons")
      .removeClass("dashicons-controls-play")
      .addClass("dashicons-controls-pause");

    var previewUrl = convergeAdmin.apiBase + "/api/voice/preview?voiceId=" +
      encodeURIComponent(voiceId) + "&voiceName=" + encodeURIComponent(voiceName || "");

    voiceAudio = new Audio(previewUrl);

    voiceAudio.oncanplaythrough = function () {
      voicePlaying = true;
      $btn.prop("disabled", false).addClass("is-playing");
      $("#voice-playing-text").show();
      voiceAudio.play();
    };

    voiceAudio.onended = function () {
      stopVoicePreview();
    };

    voiceAudio.onerror = function () {
      stopVoicePreview();
      $btn.css("border-color", "#dc3545");
      setTimeout(function () { $btn.css("border-color", ""); }, 2000);
    };

    voiceAudio.load();
  });

  // --- Job Status Action Buttons ---

  $(document).on("click", ".converge-job-status-btn", function () {
    var $btn = $(this);
    var jobId = $btn.data("job-id");
    var newStatus = $btn.data("status");

    if (!confirm("Change job status to " + newStatus + "?")) return;

    $btn.prop("disabled", true);

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_update_job",
      nonce: convergeAdmin.jobNonce,
      job_id: jobId,
      section: "basic",
      status: newStatus,
    })
      .done(function (response) {
        if (response.success) {
          location.reload();
        } else {
          alert("Error: " + (response.data || "Failed to update status"));
          $btn.prop("disabled", false);
        }
      })
      .fail(function () {
        alert("Request failed. Please try again.");
        $btn.prop("disabled", false);
      });
  });

  // --- Share Tab Switching ---

  $(document).on("click", ".converge-share-tab", function () {
    var panel = $(this).data("share");
    $(this).closest(".converge-card").find(".converge-share-tab").removeClass("is-active");
    $(this).addClass("is-active");
    $(this).closest(".converge-card").find(".converge-share-panel").removeClass("is-active");
    $(this).closest(".converge-card").find('[data-share-panel="' + panel + '"]').addClass("is-active");

    // Generate QR code on first tab switch
    if (panel === "qr" && !window._qrGenerated) {
      generateQRCode();
    }
  });

  // --- QR Code Generation (uses CDN) ---

  function generateQRCode() {
    var $container = $("#converge-qr-code");
    if (!$container.length) return;
    var url = $container.data("url");
    if (!url) return;

    // Load QRCode.js from CDN if not already loaded
    if (typeof QRCode === "undefined") {
      var script = document.createElement("script");
      script.src = "https://cdn.jsdelivr.net/npm/qrcodejs@1.0.0/qrcode.min.js";
      script.onload = function () {
        createQR($container, url);
      };
      document.head.appendChild(script);
    } else {
      createQR($container, url);
    }
  }

  function createQR($container, url) {
    $container.empty();
    new QRCode($container[0], {
      text: url,
      width: 200,
      height: 200,
      colorDark: "#1d2327",
      colorLight: "#ffffff",
    });
    window._qrGenerated = true;
  }

  // Download QR as PNG
  $(document).on("click", "#converge-download-qr", function () {
    var canvas = $("#converge-qr-code canvas")[0];
    if (!canvas) return;
    var link = document.createElement("a");
    link.download = "converge-qr-code.png";
    link.href = canvas.toDataURL("image/png");
    link.click();
  });

  // --- Candidate Status Filtering ---

  $(document).on("click", ".converge-filter-pill", function () {
    var filter = $(this).data("filter");
    $(".converge-filter-pill").removeClass("is-active");
    $(this).addClass("is-active");

    $(".converge-candidate-card").each(function () {
      if (filter === "all") {
        $(this).show();
      } else {
        $(this).toggle($(this).data("review-status") === filter);
      }
    });
  });

  // --- Candidate Sorting ---

  $("#converge-candidate-sort").on("change", function () {
    var sort = $(this).val();
    var $list = $("#converge-candidates-list");
    var $cards = $list.find(".converge-candidate-card").toArray();

    $cards.sort(function (a, b) {
      if (sort === "newest") {
        return new Date($(b).data("date")) - new Date($(a).data("date"));
      } else if (sort === "oldest") {
        return new Date($(a).data("date")) - new Date($(b).data("date"));
      } else if (sort === "name") {
        return ($(a).data("name") || "").localeCompare($(b).data("name") || "");
      }
      return 0;
    });

    $.each($cards, function (i, el) {
      $list.append(el);
    });
  });

  // --- Review Status Actions (Shortlist / Archive) ---

  $(document).on("click", ".converge-review-action", function () {
    var $btn = $(this);
    var conversationId = $btn.data("conversation-id");
    var action = $btn.data("action");

    $btn.prop("disabled", true);

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_update_review",
      nonce: convergeAdmin.jobNonce,
      conversation_id: conversationId,
      review_status: action,
    })
      .done(function (response) {
        if (response.success) {
          // Update the card's data attribute and reload to refresh UI
          location.reload();
        } else {
          alert("Error: " + (response.data || "Failed to update"));
          $btn.prop("disabled", false);
        }
      })
      .fail(function () {
        alert("Request failed.");
        $btn.prop("disabled", false);
      });
  });

  // --- Conversation Detail: Insights Polling ---

  (function () {
    var $analyzing = $(".converge-analyzing-card[data-conversation-id]");
    if (!$analyzing.length) return;

    var convId = $analyzing.data("conversation-id");
    var pollInterval = setInterval(function () {
      $.get(convergeAdmin.ajaxUrl, {
        action: "converge_check_insights",
        nonce: convergeAdmin.jobNonce,
        conversation_id: convId,
      })
        .done(function (response) {
          if (response.success && response.data && response.data.status !== "pending" && response.data.status !== "processing") {
            clearInterval(pollInterval);
            location.reload();
          }
        });
    }, 10000);
  })();

  // --- Analytics: Chart.js Integration ---

  (function () {
    var $trendChart = $("#converge-trend-chart");
    var $statusChart = $("#converge-status-chart");
    var $channelChart = $("#converge-channel-chart");

    if (!$trendChart.length && !$statusChart.length) return;

    // Load Chart.js from CDN
    if (typeof Chart === "undefined") {
      var script = document.createElement("script");
      script.src = "https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js";
      script.onload = function () {
        initCharts();
      };
      document.head.appendChild(script);
    } else {
      initCharts();
    }

    function initCharts() {
      // Trend chart
      if ($trendChart.length) {
        var trendData = $trendChart.data("trend") || [];
        new Chart($trendChart[0], {
          type: "line",
          data: {
            labels: trendData.map(function (d) { return d.date; }),
            datasets: [{
              label: "Conversations",
              data: trendData.map(function (d) { return d.count; }),
              borderColor: "#0073aa",
              backgroundColor: "rgba(0, 115, 170, 0.1)",
              fill: true,
              tension: 0.3,
            }],
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: { legend: { display: false } },
            scales: { y: { beginAtZero: true, ticks: { stepSize: 1 } } },
          },
        });
      }

      // Status chart
      if ($statusChart.length) {
        var statusData = $statusChart.data("status") || [];
        var statusColors = {
          completed: "#28a745",
          in_progress: "#0073aa",
          pending: "#f0ad4e",
          abandoned: "#dc3545",
          failed: "#dc3545",
        };
        new Chart($statusChart[0], {
          type: "doughnut",
          data: {
            labels: statusData.map(function (d) { return d.status.replace("_", " "); }),
            datasets: [{
              data: statusData.map(function (d) { return d.count; }),
              backgroundColor: statusData.map(function (d) { return statusColors[d.status] || "#c3c4c7"; }),
            }],
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: { legend: { position: "bottom" } },
          },
        });
      }

      // Channel chart
      if ($channelChart.length) {
        var channelData = $channelChart.data("channels") || [];
        new Chart($channelChart[0], {
          type: "bar",
          data: {
            labels: channelData.map(function (d) { return d.channel; }),
            datasets: [{
              label: "Conversations",
              data: channelData.map(function (d) { return d.count; }),
              backgroundColor: "#0073aa",
            }],
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: { legend: { display: false } },
            scales: { y: { beginAtZero: true, ticks: { stepSize: 1 } } },
          },
        });
      }
    }
  })();

  // --- Analytics Enhanced Sections Expand/Collapse ---

  $(document).on("click", ".converge-analytics-section-header", function () {
    $(this).closest(".converge-analytics-section").toggleClass("is-expanded");
  });

  // --- Analytics CSV Export ---

  $(document).on("click", "#converge-export-csv", function () {
    var $btn = $(this);
    $btn.prop("disabled", true).text("Exporting...");

    $.get(convergeAdmin.ajaxUrl, {
      action: "converge_export_analytics",
      nonce: convergeAdmin.jobNonce,
    })
      .done(function (response) {
        if (response.success && response.data && response.data.csv) {
          var blob = new Blob([response.data.csv], { type: "text/csv" });
          var url = URL.createObjectURL(blob);
          var link = document.createElement("a");
          link.href = url;
          link.download = "converge-analytics-" + new Date().toISOString().split("T")[0] + ".csv";
          link.click();
          URL.revokeObjectURL(url);
        }
      })
      .always(function () {
        $btn.prop("disabled", false).text("Export CSV");
      });
  });

  // --- Settings Page Handlers ---

  // Sync Now button
  $("#converge-sync-now").on("click", function () {
    var $btn = $(this);
    var $status = $("#converge-sync-status");

    $btn.prop("disabled", true).text("Syncing...");
    $status.text("");

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_sync_now",
      nonce: convergeAdmin.syncNonce,
    })
      .done(function (response) {
        if (response.success) {
          var d = response.data;
          $status.text(d.synced + " of " + d.total + " jobs synced.");
          if (d.failed > 0) {
            $status.append(" " + d.failed + " failed.");
          }
        } else {
          $status.text("Sync failed: " + (response.data || "Unknown error"));
        }
      })
      .fail(function () {
        $status.text("Request failed. Please try again.");
      })
      .always(function () {
        $btn.prop("disabled", false).text("Sync Now");
      });
  });

  // Sync enabled toggle
  $("#converge-sync-enabled").on("change", function () {
    var enabled = $(this).is(":checked") ? "1" : "0";
    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_save_setting",
      nonce: $(this).data("nonce"),
      key: "converge_sync_enabled",
      value: enabled,
    });
  });

  // Widget mode change
  $("#converge-widget-mode").on("change", function () {
    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_save_setting",
      nonce: $(this).data("nonce"),
      key: "converge_widget_mode",
      value: $(this).val(),
    });
  });

  // Sync frequency change
  $("#converge-sync-frequency").on("change", function () {
    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_save_setting",
      nonce: $("#converge-sync-enabled").data("nonce"),
      key: "converge_sync_frequency",
      value: $(this).val(),
    });
  });

  // Disconnect button
  $("#converge-disconnect").on("click", function () {
    if (
      !confirm(
        "Are you sure you want to disconnect from Converge? Job sync will stop."
      )
    ) {
      return;
    }

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_disconnect",
      nonce: convergeAdmin.nonce,
    }).done(function (response) {
      if (response.success) {
        location.reload();
      }
    });
  });

  // --- Team Management ---

  $(document).on("click", "#converge-invite-member", function () {
    var email = $("#converge-invite-email").val();
    var role = $("#converge-invite-role").val();
    var $btn = $(this);
    var $status = $("#converge-invite-status");

    if (!email) {
      $status.text("Enter an email address.").addClass("converge-save-error");
      return;
    }

    $btn.prop("disabled", true);
    $status.text("Sending invite...").removeClass("converge-save-error").addClass("converge-save-active");

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_invite_member",
      nonce: convergeAdmin.jobNonce,
      email: email,
      role: role,
    })
      .done(function (response) {
        if (response.success) {
          $status.text("Invite sent!").removeClass("converge-save-active");
          $("#converge-invite-email").val("");
          setTimeout(function () { location.reload(); }, 1500);
        } else {
          $status.text("Error: " + (response.data || "Failed")).addClass("converge-save-error").removeClass("converge-save-active");
        }
      })
      .fail(function () {
        $status.text("Request failed.").addClass("converge-save-error").removeClass("converge-save-active");
      })
      .always(function () {
        $btn.prop("disabled", false);
      });
  });

  $(document).on("click", ".converge-remove-member", function () {
    if (!confirm("Remove this team member?")) return;
    var $btn = $(this);
    var userId = $btn.data("user-id");

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_remove_member",
      nonce: convergeAdmin.jobNonce,
      user_id: userId,
    })
      .done(function (response) {
        if (response.success) {
          location.reload();
        } else {
          alert("Error: " + (response.data || "Failed"));
        }
      });
  });

  // --- Billing Actions ---

  $(document).on("click", "#converge-upgrade-pro", function () {
    var $btn = $(this);
    $btn.prop("disabled", true).text("Redirecting...");

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_create_checkout",
      nonce: convergeAdmin.jobNonce,
    })
      .done(function (response) {
        if (response.success && response.data && response.data.url) {
          window.location.href = response.data.url;
        } else {
          alert("Error: " + (response.data || "Failed to create checkout"));
          $btn.prop("disabled", false).text("Upgrade to Pro");
        }
      })
      .fail(function () {
        alert("Request failed.");
        $btn.prop("disabled", false).text("Upgrade to Pro");
      });
  });

  $(document).on("click", "#converge-manage-billing", function () {
    var $btn = $(this);
    $btn.prop("disabled", true);

    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_billing_portal",
      nonce: convergeAdmin.jobNonce,
    })
      .done(function (response) {
        if (response.success && response.data && response.data.url) {
          window.open(response.data.url, "_blank");
        } else {
          alert("Error: " + (response.data || "Failed"));
        }
      })
      .always(function () {
        $btn.prop("disabled", false);
      });
  });

  // ===== Team Members (Settings Page) =====

  if ($("#converge-team-list").length) {
    loadTeamMembers();
  }

  function loadTeamMembers() {
    $.get(convergeAdmin.ajaxUrl, {
      action: "converge_get_team_members",
      nonce: convergeAdmin.jobNonce,
    })
      .done(function (response) {
        if (response.success && response.data && response.data.members) {
          renderTeamTable(response.data.members);
        } else {
          $("#converge-team-list").html(
            '<p class="description">No team members found.</p>'
          );
        }
      })
      .fail(function () {
        $("#converge-team-list").html(
          '<p class="description">Unable to load team members. Please check your connection.</p>'
        );
      });
  }

  function renderTeamTable(members) {
    if (!members.length) {
      $("#converge-team-list").html(
        '<p class="description">No team members found.</p>'
      );
      return;
    }

    var html =
      '<table class="converge-team-table"><thead><tr>' +
      "<th>Name</th><th>Email</th><th>Role</th><th>Joined</th><th></th>" +
      "</tr></thead><tbody>";

    members.forEach(function (m) {
      var joined = m.joinedAt
        ? new Date(m.joinedAt).toLocaleDateString()
        : "—";
      var roleBadgeClass = m.role === "admin" ? "converge-role-badge-admin" : "converge-role-badge-member";
      var roleLabel = m.role === "admin" ? "Admin" : "Member";
      html +=
        "<tr>" +
        "<td>" +
        escapeHtml(m.name) +
        "</td>" +
        "<td>" +
        escapeHtml(m.email) +
        "</td>" +
        '<td><span class="converge-role-badge ' + roleBadgeClass + '">' + roleLabel + '</span></td>' +
        "<td>" +
        joined +
        "</td>" +
        '<td><button type="button" class="converge-team-remove" data-user-id="' +
        m.id +
        '">Remove</button></td>' +
        "</tr>";
    });

    html += "</tbody></table>";
    $("#converge-team-list").html(html);
  }

  function escapeHtml(str) {
    if (!str) return "";
    var div = document.createElement("div");
    div.appendChild(document.createTextNode(str));
    return div.innerHTML;
  }

  // ===== AI Defaults (Settings Page) =====

  if ($("#converge-ai-defaults").length) {
    loadAIConfig();
  }

  var aiConfigData = null; // Store full AI config response

  function loadAIConfig() {
    $.get(convergeAdmin.ajaxUrl, {
      action: "converge_get_ai_config",
      nonce: convergeAdmin.jobNonce,
    })
      .done(function (response) {
        if (response.success && response.data) {
          aiConfigData = response.data;
          renderAIConfig(response.data);
        } else {
          $("#converge-ai-defaults").html(
            '<p class="description">Unable to load AI configuration.</p>'
          );
        }
      })
      .fail(function () {
        $("#converge-ai-defaults").html(
          '<p class="description">Unable to load AI configuration. Please check your connection.</p>'
        );
      });
  }

  function renderAIConfig(data) {
    var config = data.config || {};
    var screening = data.screeningDefaults || {};
    var convConfig = data.conversationConfig || {};
    var customGreeting = data.customGreeting || "";
    var useCustomGreeting = data.useCustomGreeting || false;

    var personality = config.default_personality || "professional";
    var pace = config.default_pace || "moderate";
    var depth = config.default_probing_depth || 3;
    var prosody = config.prosody_analysis_enabled !== false;

    // Default screening fields
    var screeningFields = [
      { key: "collect_availability", label: "Availability / Start Date", desc: "When can the candidate start?" },
      { key: "collect_salary_expectations", label: "Salary Expectations", desc: "Expected compensation range." },
      { key: "collect_work_authorization", label: "Work Authorization", desc: "Legal authorization to work." },
      { key: "collect_notice_period", label: "Notice Period", desc: "Current notice period if employed." },
      { key: "collect_location", label: "Location", desc: "Candidate's current location." },
      { key: "collect_relocation", label: "Relocation Willingness", desc: "Open to relocating for the role." },
      { key: "collect_employment_status", label: "Employment Status", desc: "Current employment situation." },
    ];

    // Default conversation sections
    var bgConfig = convConfig.background || { enabled: true, depth: 3, topics: { work_history: true, education: true, skills: true, achievements: true } };
    var motConfig = convConfig.motivation || { enabled: true, depth: 3, topics: { role_interest: true, company_interest: true, career_goals: true, values_alignment: true } };
    var cfConfig = convConfig.culture_fit || { enabled: false, depth: 2, topics: { work_style: true, team_preference: true, management_style: true, conflict_resolution: true } };

    var defaultGreeting = "Hi there! Thanks for taking some time to chat with us. I'm here to learn about your background and answer any questions you might have. We'll go at your pace and make sure we cover everything. Before we dive in, could you tell me your name?";

    // Build tabs
    var html = '<div class="converge-ai-tabs">' +
      '<button class="converge-ai-tab is-active" data-tab="screening">Screening</button>' +
      '<button class="converge-ai-tab" data-tab="conversation">Conversation</button>' +
      '<button class="converge-ai-tab" data-tab="ai-style">AI Style</button>' +
      '<button class="converge-ai-tab" data-tab="greeting">Greeting</button>' +
    '</div>';

    // Tab 1: Screening
    html += '<div class="converge-ai-tab-panel is-active" data-tab="screening">' +
      '<p class="description" style="margin-bottom:12px;">Default screening data to collect during conversations. Individual jobs can override these.</p>' +
      '<div class="converge-screening-grid">';
    for (var i = 0; i < screeningFields.length; i++) {
      var f = screeningFields[i];
      var checked = screening[f.key] !== false; // default true
      html += '<div class="converge-toggle-row">' +
        '<div class="converge-toggle-label">' +
          '<strong>' + f.label + '</strong>' +
          '<span class="description">' + f.desc + '</span>' +
        '</div>' +
        '<label class="converge-toggle">' +
          '<input type="checkbox" class="converge-screening-toggle" data-key="' + f.key + '"' + (checked ? ' checked' : '') + '>' +
          '<span class="converge-toggle-slider"></span>' +
        '</label>' +
      '</div>';
    }
    html += '</div>' +
      '<div style="margin-top:12px;">' +
        '<button type="button" class="button button-primary" id="converge-save-screening">Save Screening Defaults</button>' +
        '<span id="converge-screening-status" class="description" style="margin-left:8px;"></span>' +
      '</div>' +
    '</div>';

    // Tab 2: Conversation
    html += '<div class="converge-ai-tab-panel" data-tab="conversation">' +
      '<p class="description" style="margin-bottom:12px;">Configure which conversation topics the AI explores with candidates.</p>';

    var convSections = [
      { key: "background", label: "Background Exploration", config: bgConfig, topics: [
        { key: "work_history", label: "Work History" }, { key: "education", label: "Education" },
        { key: "skills", label: "Skills & Expertise" }, { key: "achievements", label: "Key Achievements" }
      ]},
      { key: "motivation", label: "Motivation & Interest", config: motConfig, topics: [
        { key: "role_interest", label: "Role Interest" }, { key: "company_interest", label: "Company Interest" },
        { key: "career_goals", label: "Career Goals" }, { key: "values_alignment", label: "Values Alignment" }
      ]},
      { key: "culture_fit", label: "Culture Fit", config: cfConfig, topics: [
        { key: "work_style", label: "Work Style" }, { key: "team_preference", label: "Team Preference" },
        { key: "management_style", label: "Management Style" }, { key: "conflict_resolution", label: "Conflict Resolution" }
      ]}
    ];

    for (var s = 0; s < convSections.length; s++) {
      var sec = convSections[s];
      var secEnabled = sec.config.enabled !== false;
      var secDepth = sec.config.depth || 3;
      html += '<div class="converge-conv-card' + (secEnabled ? ' is-open' : '') + '" data-section="' + sec.key + '">' +
        '<div class="converge-conv-card-header">' +
          '<div style="display:flex;align-items:center;gap:8px;">' +
            '<label class="converge-toggle" style="margin:0;" onclick="event.stopPropagation();">' +
              '<input type="checkbox" class="converge-conv-enabled" data-section="' + sec.key + '"' + (secEnabled ? ' checked' : '') + '>' +
              '<span class="converge-toggle-slider"></span>' +
            '</label>' +
            '<strong>' + sec.label + '</strong>' +
          '</div>' +
          '<span class="converge-conv-card-chevron dashicons dashicons-arrow-down-alt2"></span>' +
        '</div>' +
        '<div class="converge-conv-card-body">' +
          '<div class="converge-conv-depth-wrap">' +
            '<label>Depth:</label>' +
            '<input type="range" class="converge-conv-depth" data-section="' + sec.key + '" min="1" max="5" value="' + secDepth + '">' +
            '<span class="converge-conv-depth-val">' + secDepth + '</span>' +
          '</div>';
      for (var t = 0; t < sec.topics.length; t++) {
        var topic = sec.topics[t];
        var topicEnabled = sec.config.topics && sec.config.topics[topic.key] !== false;
        html += '<div class="converge-toggle-row" style="padding:6px 0;">' +
          '<div class="converge-toggle-label"><strong>' + topic.label + '</strong></div>' +
          '<label class="converge-toggle">' +
            '<input type="checkbox" class="converge-conv-topic" data-section="' + sec.key + '" data-topic="' + topic.key + '"' + (topicEnabled ? ' checked' : '') + '>' +
            '<span class="converge-toggle-slider"></span>' +
          '</label>' +
        '</div>';
      }
      html += '</div></div>';
    }

    html += '<div style="margin-top:12px;">' +
      '<button type="button" class="button button-primary" id="converge-save-conversation">Save Conversation Config</button>' +
      '<span id="converge-conversation-status" class="description" style="margin-left:8px;"></span>' +
    '</div></div>';

    // Voice options
    var voices = [
      { id: "kora-voice", name: "KORA", label: "Kora", desc: "Warm and engaging. Works well for all roles.", gender: "F", recommended: true },
      { id: "ito-voice", name: "ITO", label: "Ito", desc: "Clear and professional. Good for technical roles.", gender: "M" },
      { id: "serene-assistant", name: "Serene Assistant", label: "Sofia", desc: "Calm and reassuring. Ideal for thoughtful conversations.", gender: "F" },
      { id: "male-english-actor", name: "Male English Actor", label: "James", desc: "Polished and articulate. Great for professional services.", gender: "M" },
      { id: "expressive-girl", name: "Expressive Girl", label: "Ella", desc: "Warm and expressive. Great for engaging conversations.", gender: "F" },
      { id: "174ddb73-a079-48a7-ab99-8f03b4f9902a", name: "Marcus", label: "Marcus", desc: "Confident and professional. Great for corporate roles.", gender: "M" },
    ];
    var currentVoiceId = config.hume_voice_id || "kora-voice";
    var currentVoiceName = config.hume_voice_name || "KORA";

    // Tab 3: AI Style (existing + voice)
    html += '<div class="converge-ai-tab-panel" data-tab="ai-style">' +
      '<div class="converge-ai-config-form">' +
      '<div class="converge-ai-field">' +
        '<label><strong>AI Voice</strong></label>' +
        '<p class="description" style="margin-bottom:8px;">Select the voice your AI interviewer will use.</p>' +
        '<div class="converge-voice-grid">';
    for (var v = 0; v < voices.length; v++) {
      var voice = voices[v];
      var isSelected = voice.id === currentVoiceId;
      html += '<div class="converge-voice-option' + (isSelected ? ' is-selected' : '') + '" data-voice-id="' + voice.id + '" data-voice-name="' + voice.name + '">' +
        '<div style="display:flex;align-items:center;justify-content:space-between;">' +
          '<div>' +
            '<strong>' + voice.label + '</strong>' +
            (voice.recommended ? ' <span class="converge-voice-rec">Recommended</span>' : '') +
            '<br><span class="description" style="font-size:12px;">' + voice.desc + '</span>' +
          '</div>' +
          '<button type="button" class="button converge-voice-preview" data-voice-id="' + voice.id + '" data-voice-name="' + voice.name + '" title="Preview voice">' +
            '<span class="dashicons dashicons-controls-play" style="margin-top:3px;"></span>' +
          '</button>' +
        '</div>' +
      '</div>';
    }
    html += '</div>' +
      '</div>' +
      '<div class="converge-ai-field">' +
        '<label>Personality</label>' +
        '<select id="converge-ai-personality" class="regular-text">' +
          '<option value="friendly"' + (personality === "friendly" ? " selected" : "") + '>Friendly</option>' +
          '<option value="professional"' + (personality === "professional" ? " selected" : "") + '>Professional</option>' +
          '<option value="casual"' + (personality === "casual" ? " selected" : "") + '>Casual</option>' +
          '<option value="formal"' + (personality === "formal" ? " selected" : "") + '>Formal</option>' +
        '</select>' +
        '<p class="description">Sets the tone of voice for AI conversations.</p>' +
      '</div>' +
      '<div class="converge-ai-field">' +
        '<label>Pace</label>' +
        '<select id="converge-ai-pace" class="regular-text">' +
          '<option value="relaxed"' + (pace === "relaxed" ? " selected" : "") + '>Relaxed</option>' +
          '<option value="moderate"' + (pace === "moderate" ? " selected" : "") + '>Moderate</option>' +
          '<option value="brisk"' + (pace === "brisk" ? " selected" : "") + '>Brisk</option>' +
        '</select>' +
        '<p class="description">Controls the conversation speed.</p>' +
      '</div>' +
      '<div class="converge-ai-field">' +
        '<label>Probing Depth</label>' +
        '<div class="converge-range-wrap">' +
          '<input type="range" id="converge-ai-depth" min="1" max="5" value="' + depth + '">' +
          '<span class="converge-range-value" id="converge-ai-depth-val">' + depth + '</span>' +
        '</div>' +
        '<div class="converge-range-labels"><span>Surface-level</span><span>Deep dive</span></div>' +
        '<p class="description">How deeply the AI probes into candidate responses.</p>' +
      '</div>' +
      '<div class="converge-ai-field">' +
        '<div class="converge-toggle-row" style="border-bottom:none;padding:0;">' +
          '<div class="converge-toggle-label">' +
            '<strong>Voice Engagement Analysis</strong>' +
            '<span class="description">Analyze prosody signals (tone, pace, engagement) during conversations.</span>' +
          '</div>' +
          '<label class="converge-toggle">' +
            '<input type="checkbox" id="converge-ai-prosody"' + (prosody ? " checked" : "") + '>' +
            '<span class="converge-toggle-slider"></span>' +
          '</label>' +
        '</div>' +
      '</div>' +
      '<div style="margin-top:12px;">' +
        '<button type="button" class="button button-primary" id="converge-save-ai-defaults">Save AI Style</button>' +
        '<span id="converge-ai-save-status" class="description" style="margin-left:8px;"></span>' +
      '</div>' +
    '</div></div>';

    // Tab 4: Greeting
    html += '<div class="converge-ai-tab-panel" data-tab="greeting">' +
      '<div class="converge-toggle-row" style="border-bottom:none;padding:0;margin-bottom:12px;">' +
        '<div class="converge-toggle-label">' +
          '<strong>Use Custom Greeting</strong>' +
          '<span class="description">Override the default greeting with your own.</span>' +
        '</div>' +
        '<label class="converge-toggle">' +
          '<input type="checkbox" id="converge-greeting-toggle"' + (useCustomGreeting ? ' checked' : '') + '>' +
          '<span class="converge-toggle-slider"></span>' +
        '</label>' +
      '</div>' +
      '<div id="converge-greeting-default-wrap"' + (useCustomGreeting ? ' style="display:none;"' : '') + '>' +
        '<label><strong>Default Greeting</strong></label>' +
        '<div class="converge-greeting-default">' + defaultGreeting + '</div>' +
      '</div>' +
      '<div id="converge-greeting-custom-wrap"' + (!useCustomGreeting ? ' style="display:none;"' : '') + '>' +
        '<label for="converge-greeting-text"><strong>Custom Greeting</strong></label>' +
        '<textarea id="converge-greeting-text" class="large-text" rows="4" placeholder="Enter your custom greeting...">' + (customGreeting || '') + '</textarea>' +
        '<p class="description">The AI will use this greeting at the start of every conversation. It should end with an open-ended question.</p>' +
      '</div>' +
      '<div style="margin-top:12px;">' +
        '<button type="button" class="button button-primary" id="converge-save-greeting">Save Greeting</button>' +
        '<span id="converge-greeting-status" class="description" style="margin-left:8px;"></span>' +
      '</div>' +
    '</div>';

    $("#converge-ai-defaults").html(html);

    // Tab switching
    $(".converge-ai-tab").on("click", function () {
      var tab = $(this).data("tab");
      $(".converge-ai-tab").removeClass("is-active");
      $(this).addClass("is-active");
      $(".converge-ai-tab-panel").removeClass("is-active");
      $('.converge-ai-tab-panel[data-tab="' + tab + '"]').addClass("is-active");
    });

    // Conversation card collapsible
    $(".converge-conv-card-header").on("click", function () {
      $(this).closest(".converge-conv-card").toggleClass("is-open");
    });

    // Conversation depth slider display
    $(".converge-conv-depth").on("input", function () {
      $(this).siblings(".converge-conv-depth-val").text($(this).val());
    });

    // Greeting toggle
    $("#converge-greeting-toggle").on("change", function () {
      if ($(this).is(":checked")) {
        $("#converge-greeting-default-wrap").hide();
        $("#converge-greeting-custom-wrap").show();
      } else {
        $("#converge-greeting-default-wrap").show();
        $("#converge-greeting-custom-wrap").hide();
      }
    });

    // Voice selection
    $(".converge-voice-option").on("click", function (e) {
      if ($(e.target).closest(".converge-voice-preview").length) return; // Don't select on preview click
      $(".converge-voice-option").removeClass("is-selected");
      $(this).addClass("is-selected");
    });

    // Voice preview
    var previewAudio = null;
    $(".converge-voice-preview").on("click", function () {
      var $btn = $(this);
      var voiceId = $btn.data("voice-id");
      var voiceName = $btn.data("voice-name");

      // Stop any playing preview
      if (previewAudio) {
        previewAudio.pause();
        previewAudio = null;
        $(".converge-voice-preview .dashicons").removeClass("dashicons-controls-pause").addClass("dashicons-controls-play");
      }

      // If clicking same button that was playing, just stop
      if ($btn.data("playing")) {
        $btn.data("playing", false);
        return;
      }

      $(".converge-voice-preview").data("playing", false);
      $btn.data("playing", true);
      $btn.find(".dashicons").removeClass("dashicons-controls-play").addClass("dashicons-update");

      var previewUrl = convergeAdmin.apiBase + "/api/voice/preview?voiceId=" + encodeURIComponent(voiceId) + "&voiceName=" + encodeURIComponent(voiceName);
      previewAudio = new Audio(previewUrl);
      previewAudio.oncanplaythrough = function () {
        $btn.find(".dashicons").removeClass("dashicons-update").addClass("dashicons-controls-pause");
        previewAudio.play();
      };
      previewAudio.onended = function () {
        $btn.find(".dashicons").removeClass("dashicons-controls-pause").addClass("dashicons-controls-play");
        $btn.data("playing", false);
        previewAudio = null;
      };
      previewAudio.onerror = function () {
        $btn.find(".dashicons").removeClass("dashicons-update").addClass("dashicons-controls-play");
        $btn.data("playing", false);
        previewAudio = null;
      };
    });

    // AI Style depth slider
    $("#converge-ai-depth").on("input", function () {
      $("#converge-ai-depth-val").text($(this).val());
    });

    // Save Screening Defaults
    $("#converge-save-screening").on("click", function () {
      var $btn = $(this), $status = $("#converge-screening-status");
      $btn.prop("disabled", true);
      $status.text("Saving...");
      var screeningData = {};
      $(".converge-screening-toggle").each(function () {
        screeningData[$(this).data("key")] = $(this).is(":checked");
      });
      $.post(convergeAdmin.ajaxUrl, {
        action: "converge_save_ai_config",
        nonce: convergeAdmin.jobNonce,
        screening_defaults: JSON.stringify(screeningData),
      }).done(function (r) {
        if (r.success) {
          $status.text("Saved!").css("color", "#16a34a");
          setTimeout(function () { $status.text(""); }, 2000);
        } else { $status.text("Failed.").css("color", "#dc3545"); }
      }).fail(function () { $status.text("Failed to save.").css("color", "#dc3545"); })
        .always(function () { $btn.prop("disabled", false); });
    });

    // Save Conversation Config
    $("#converge-save-conversation").on("click", function () {
      var $btn = $(this), $status = $("#converge-conversation-status");
      $btn.prop("disabled", true);
      $status.text("Saving...");
      var convData = {};
      $(".converge-conv-card").each(function () {
        var section = $(this).data("section");
        var topics = {};
        $(this).find(".converge-conv-topic").each(function () {
          topics[$(this).data("topic")] = $(this).is(":checked");
        });
        convData[section] = {
          enabled: $(this).find(".converge-conv-enabled").is(":checked"),
          depth: parseInt($(this).find(".converge-conv-depth").val()) || 3,
          topics: topics,
        };
      });
      $.post(convergeAdmin.ajaxUrl, {
        action: "converge_save_ai_config",
        nonce: convergeAdmin.jobNonce,
        conversation_config: JSON.stringify(convData),
      }).done(function (r) {
        if (r.success) {
          $status.text("Saved!").css("color", "#16a34a");
          setTimeout(function () { $status.text(""); }, 2000);
        } else { $status.text("Failed.").css("color", "#dc3545"); }
      }).fail(function () { $status.text("Failed to save.").css("color", "#dc3545"); })
        .always(function () { $btn.prop("disabled", false); });
    });

    // Save AI Style
    $("#converge-save-ai-defaults").on("click", function () {
      var $btn = $(this), $status = $("#converge-ai-save-status");
      $btn.prop("disabled", true);
      $status.text("Saving...");
      var $selectedVoice = $(".converge-voice-option.is-selected");
      var voiceId = $selectedVoice.data("voice-id") || "";
      var voiceName = $selectedVoice.data("voice-name") || "";
      $.post(convergeAdmin.ajaxUrl, {
        action: "converge_save_ai_config",
        nonce: convergeAdmin.jobNonce,
        default_personality: $("#converge-ai-personality").val(),
        default_pace: $("#converge-ai-pace").val(),
        default_probing_depth: $("#converge-ai-depth").val(),
        prosody_analysis_enabled: $("#converge-ai-prosody").is(":checked") ? "true" : "false",
        hume_voice_id: voiceId,
        hume_voice_name: voiceName,
      }).done(function (r) {
        if (r.success) {
          $status.text("Saved!").css("color", "#16a34a");
          setTimeout(function () { $status.text(""); }, 2000);
        } else { $status.text("Failed.").css("color", "#dc3545"); }
      }).fail(function () { $status.text("Failed to save.").css("color", "#dc3545"); })
        .always(function () { $btn.prop("disabled", false); });
    });

    // Save Greeting
    $("#converge-save-greeting").on("click", function () {
      var $btn = $(this), $status = $("#converge-greeting-status");
      $btn.prop("disabled", true);
      $status.text("Saving...");
      $.post(convergeAdmin.ajaxUrl, {
        action: "converge_save_ai_config",
        nonce: convergeAdmin.jobNonce,
        use_custom_greeting: $("#converge-greeting-toggle").is(":checked") ? "true" : "false",
        custom_greeting: $("#converge-greeting-text").val(),
      }).done(function (r) {
        if (r.success) {
          $status.text("Saved!").css("color", "#16a34a");
          setTimeout(function () { $status.text(""); }, 2000);
        } else { $status.text("Failed.").css("color", "#dc3545"); }
      }).fail(function () { $status.text("Failed to save.").css("color", "#dc3545"); })
        .always(function () { $btn.prop("disabled", false); });
    });
  }

  // ===== Job Automation (Settings Page) =====

  $("#converge-save-automation").on("click", function () {
    var $btn = $(this), $status = $("#converge-automation-status");
    $btn.prop("disabled", true);
    $status.text("Saving...");
    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_save_org_settings",
      nonce: convergeAdmin.jobNonce,
      widget_auto_create: $("#converge-auto-create").is(":checked") ? "true" : "false",
      widget_auto_publish: $("#converge-auto-publish").is(":checked") ? "true" : "false",
    }).done(function (r) {
      if (r.success) {
        $status.text("Saved!").css("color", "#16a34a");
        setTimeout(function () { $status.text(""); }, 2000);
      } else { $status.text("Failed.").css("color", "#dc3545"); }
    }).fail(function () { $status.text("Failed to save.").css("color", "#dc3545"); })
      .always(function () { $btn.prop("disabled", false); });
  });

  // ===== Candidate Portal (Settings Page) =====

  $("#converge-save-portal").on("click", function () {
    var $btn = $(this), $status = $("#converge-portal-save-status");
    $btn.prop("disabled", true);
    $status.text("Saving...");
    var portal = {
      show_application_status: $("#converge-portal-status").is(":checked"),
      show_shortlisted: $("#converge-portal-shortlisted").is(":checked"),
      show_archived: $("#converge-portal-archived").is(":checked"),
    };
    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_save_org_settings",
      nonce: convergeAdmin.jobNonce,
      candidate_portal_settings: JSON.stringify(portal),
    }).done(function (r) {
      if (r.success) {
        $status.text("Saved!").css("color", "#16a34a");
        setTimeout(function () { $status.text(""); }, 2000);
      } else { $status.text("Failed.").css("color", "#dc3545"); }
    }).fail(function () { $status.text("Failed to save.").css("color", "#dc3545"); })
      .always(function () { $btn.prop("disabled", false); });
  });

  // ===== Data Retention (Settings Page) =====

  function updateRetentionLabel() {
    var days = parseInt($("#converge-retention-days").val()) || 365;
    var label = "";
    if (days >= 365) {
      var years = Math.round(days / 365 * 10) / 10;
      label = years === 1 ? "1 year" : years + " years";
    } else if (days >= 30) {
      var months = Math.round(days / 30);
      label = months === 1 ? "1 month" : months + " months";
    } else {
      label = days + " days";
    }
    $("#converge-retention-friendly").text("(" + label + ")");
  }
  if ($("#converge-retention-days").length) {
    updateRetentionLabel();
    $("#converge-retention-days").on("input", updateRetentionLabel);
  }

  $("#converge-save-retention").on("click", function () {
    var $btn = $(this), $status = $("#converge-retention-status");
    $btn.prop("disabled", true);
    $status.text("Saving...");
    $.post(convergeAdmin.ajaxUrl, {
      action: "converge_save_org_settings",
      nonce: convergeAdmin.jobNonce,
      data_retention_days: $("#converge-retention-days").val(),
    }).done(function (r) {
      if (r.success) {
        $status.text("Saved!").css("color", "#16a34a");
        setTimeout(function () { $status.text(""); }, 2000);
      } else { $status.text("Failed.").css("color", "#dc3545"); }
    }).fail(function () { $status.text("Failed to save.").css("color", "#dc3545"); })
      .always(function () { $btn.prop("disabled", false); });
  });

  // ===== Copy to Clipboard (Embed Codes) =====

  $(".converge-copy-btn").on("click", function () {
    var target = $(this).data("target");
    var text = $("#" + target).text();
    var $btn = $(this);
    if (navigator.clipboard) {
      navigator.clipboard.writeText(text).then(function () {
        $btn.text("Copied!");
        setTimeout(function () { $btn.text("Copy"); }, 2000);
      });
    } else {
      var $temp = $("<textarea>").val(text).appendTo("body").select();
      document.execCommand("copy");
      $temp.remove();
      $btn.text("Copied!");
      setTimeout(function () { $btn.text("Copy"); }, 2000);
    }
  });

  // ===== Insights Polling (Conversation Detail) =====

  var $analyzingCard = $(".converge-analyzing-card");
  if ($analyzingCard.length) {
    var pollConvId = $analyzingCard.data("conversation-id");
    var pollInterval = setInterval(function () {
      $.post(convergeAdmin.ajaxUrl, {
        action: "converge_check_insights",
        nonce: convergeAdmin.jobNonce,
        conversation_id: pollConvId,
      }).done(function (response) {
        if (response.success && response.data) {
          var status = response.data.insightsStatus;
          if (status === "completed" || status === "failed" || status === "skipped") {
            clearInterval(pollInterval);
            location.reload();
          }
        }
      });
    }, 10000);
  }

  // ===== Relative Time (Dashboard) =====

  $(".converge-relative-time").each(function () {
    var $el = $(this);
    var time = $el.data("time");
    if (time) {
      $el.text(relativeTime(time));
    }
  });

  // ===== Enhanced Analytics Loading =====

  var $chartsContainer = $("#converge-charts-container");
  if ($chartsContainer.length) {
    var analyticsDays = $chartsContainer.data("days") || 30;

    $.get(convergeAdmin.ajaxUrl, {
      action: "converge_get_enhanced_analytics",
      nonce: convergeAdmin.jobNonce,
      days: analyticsDays,
    })
      .done(function (response) {
        if (response.success && response.data) {
          initCharts(response.data);
          renderEnhancedSections(response.data);
        }
      })
      .fail(function () {
        $chartsContainer.html(
          '<p class="description">Unable to load chart data.</p>'
        );
      });
  }

  function renderEnhancedSections(data) {
    // Reliability
    var $reliabilityBody = $(
      '.converge-analytics-section[data-section="reliability"] .converge-analytics-section-body'
    );
    if ($reliabilityBody.length && data.reliability) {
      var r = data.reliability;
      var qHtml = "";
      if (r.connectionQuality) {
        Object.keys(r.connectionQuality).forEach(function (k) {
          qHtml +=
            "<span><strong>" +
            k +
            ":</strong> " +
            r.connectionQuality[k] +
            "</span>&nbsp;&nbsp;";
        });
      }
      $reliabilityBody.html(
        '<div style="display:flex;gap:24px;flex-wrap:wrap;">' +
          "<div><strong>Error Rate:</strong> " +
          (r.errorRate || 0) +
          "%</div>" +
          "<div><strong>Connection Quality:</strong> " +
          (qHtml || "N/A") +
          "</div>" +
          "<div><strong>Accommodations:</strong> " +
          (r.accommodationsCount || 0) +
          "</div>" +
          "</div>"
      );
    }

    // Assessment
    var $assessBody = $(
      '.converge-analytics-section[data-section="assessment"] .converge-analytics-section-body'
    );
    if ($assessBody.length && data.assessmentDistribution) {
      var a = data.assessmentDistribution;
      var total = a.strong + a.moderate + a.limited + a.notAssessed || 1;
      $assessBody.html(
        '<div style="display:flex;gap:16px;flex-wrap:wrap;">' +
          '<div><span style="color:#16a34a;font-weight:600;">Strong:</span> ' +
          a.strong +
          " (" +
          Math.round((a.strong / total) * 100) +
          "%)</div>" +
          '<div><span style="color:#ca8a04;font-weight:600;">Moderate:</span> ' +
          a.moderate +
          " (" +
          Math.round((a.moderate / total) * 100) +
          "%)</div>" +
          '<div><span style="color:#dc2626;font-weight:600;">Limited:</span> ' +
          a.limited +
          " (" +
          Math.round((a.limited / total) * 100) +
          "%)</div>" +
          '<div><span style="color:#6b7280;font-weight:600;">Not Assessed:</span> ' +
          a.notAssessed +
          "</div>" +
          "</div>"
      );
    }

    // Job Performance
    var $jobPerfBody = $(
      '.converge-analytics-section[data-section="jobPerformance"] .converge-analytics-section-body'
    );
    if ($jobPerfBody.length && data.jobPerformance && data.jobPerformance.length) {
      var tbl =
        '<table class="widefat striped"><thead><tr>' +
        "<th>Job</th><th>Status</th><th>Conversations</th><th>Completed</th><th>Completion Rate</th>" +
        "</tr></thead><tbody>";
      data.jobPerformance.forEach(function (j) {
        tbl +=
          "<tr><td>" +
          escapeHtml(j.title) +
          "</td><td>" +
          (j.status || "—") +
          "</td><td>" +
          j.conversations +
          "</td><td>" +
          j.completed +
          "</td><td>" +
          j.completionRate +
          "%</td></tr>";
      });
      tbl += "</tbody></table>";
      $jobPerfBody.html(tbl);
    } else if ($jobPerfBody.length) {
      $jobPerfBody.html('<p class="description">No job data available.</p>');
    }
  }

  // ===== Analytics Section Toggle =====

  $(document).on("click", ".converge-analytics-section-header", function () {
    $(this).closest(".converge-analytics-section").toggleClass("open");
  });
})(jQuery);
