<?php
/**
 * Admin menu and page rendering for Converge WordPress plugin.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Converge_Admin {

    private $auth;
    private $api;
    private $sync;

    public function __construct( Converge_Auth $auth, Converge_API $api, Converge_Sync $sync ) {
        $this->auth = $auth;
        $this->api  = $api;
        $this->sync = $sync;

        add_action( 'admin_menu', array( $this, 'register_menus' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
        add_action( 'admin_notices', array( $this, 'show_connection_notice' ) );
        add_action( 'wp_ajax_converge_save_setting', array( $this, 'ajax_save_setting' ) );
        add_action( 'wp_ajax_converge_save_company_info', array( $this, 'ajax_save_company_info' ) );
        add_action( 'wp_ajax_converge_update_job', array( $this, 'ajax_update_job' ) );
        add_action( 'wp_ajax_converge_parse_competencies', array( $this, 'ajax_parse_competencies' ) );
        add_action( 'wp_ajax_converge_generate_questions', array( $this, 'ajax_generate_questions' ) );
        add_action( 'wp_ajax_converge_update_review', array( $this, 'ajax_update_review' ) );
        add_action( 'wp_ajax_converge_check_insights', array( $this, 'ajax_check_insights' ) );
        add_action( 'wp_ajax_converge_export_analytics', array( $this, 'ajax_export_analytics' ) );
        add_action( 'wp_ajax_converge_invite_member', array( $this, 'ajax_invite_member' ) );
        add_action( 'wp_ajax_converge_remove_member', array( $this, 'ajax_remove_member' ) );
        add_action( 'wp_ajax_converge_create_checkout', array( $this, 'ajax_create_checkout' ) );
        add_action( 'wp_ajax_converge_billing_portal', array( $this, 'ajax_billing_portal' ) );
        add_action( 'wp_ajax_converge_get_team_members', array( $this, 'ajax_get_team_members' ) );
        add_action( 'wp_ajax_converge_get_ai_config', array( $this, 'ajax_get_ai_config' ) );
        add_action( 'wp_ajax_converge_save_ai_config', array( $this, 'ajax_save_ai_config' ) );
        add_action( 'wp_ajax_converge_get_enhanced_analytics', array( $this, 'ajax_get_enhanced_analytics' ) );
        add_action( 'wp_ajax_converge_save_org_settings', array( $this, 'ajax_save_org_settings' ) );
    }

    /**
     * Register wp-admin menu items.
     */
    public function register_menus() {
        // Main menu
        add_menu_page(
            __( 'Converge', 'converge-applybyvoice' ),
            __( 'Converge', 'converge-applybyvoice' ),
            'manage_options',
            'converge',
            array( $this, 'render_dashboard' ),
            'dashicons-microphone',
            30
        );

        // Dashboard (same as main)
        add_submenu_page(
            'converge',
            __( 'Dashboard', 'converge-applybyvoice' ),
            __( 'Dashboard', 'converge-applybyvoice' ),
            'manage_options',
            'converge',
            array( $this, 'render_dashboard' )
        );

        // Jobs
        add_submenu_page(
            'converge',
            __( 'Jobs', 'converge-applybyvoice' ),
            __( 'Jobs', 'converge-applybyvoice' ),
            'manage_options',
            'converge-jobs',
            array( $this, 'render_jobs' )
        );

        // Analytics
        add_submenu_page(
            'converge',
            __( 'Analytics', 'converge-applybyvoice' ),
            __( 'Analytics', 'converge-applybyvoice' ),
            'manage_options',
            'converge-analytics',
            array( $this, 'render_analytics' )
        );

        // Company
        add_submenu_page(
            'converge',
            __( 'Company', 'converge-applybyvoice' ),
            __( 'Company', 'converge-applybyvoice' ),
            'manage_options',
            'converge-company',
            array( $this, 'render_company' )
        );

        // Job Detail (hidden from menu — accessed via Jobs table link)
        add_submenu_page(
            null, // Hidden from menu
            __( 'Job Detail', 'converge-applybyvoice' ),
            __( 'Job Detail', 'converge-applybyvoice' ),
            'manage_options',
            'converge-job-detail',
            array( $this, 'render_job_detail' )
        );

        // Conversation Detail (hidden from menu — accessed via Job Detail table link)
        add_submenu_page(
            null, // Hidden from menu
            __( 'Candidate Review', 'converge-applybyvoice' ),
            __( 'Candidate Review', 'converge-applybyvoice' ),
            'manage_options',
            'converge-conversation',
            array( $this, 'render_conversation' )
        );

        // Settings (native)
        add_submenu_page(
            'converge',
            __( 'Settings', 'converge-applybyvoice' ),
            __( 'Settings', 'converge-applybyvoice' ),
            'manage_options',
            'converge-settings',
            array( $this, 'render_settings' )
        );
    }

    /**
     * Enqueue admin CSS and JS on Converge pages only.
     */
    public function enqueue_assets( $hook_suffix ) {
        // Only load on our plugin pages
        if ( strpos( $hook_suffix, 'converge' ) === false ) {
            return;
        }

        wp_enqueue_style(
            'converge-admin',
            CONVERGE_PLUGIN_URL . 'assets/css/converge-admin.css',
            array(),
            CONVERGE_VERSION
        );

        wp_enqueue_script(
            'converge-admin',
            CONVERGE_PLUGIN_URL . 'assets/js/converge-admin.js',
            array( 'jquery' ),
            CONVERGE_VERSION,
            true
        );

        wp_enqueue_script(
            'converge-auth',
            CONVERGE_PLUGIN_URL . 'assets/js/converge-auth.js',
            array( 'jquery' ),
            CONVERGE_VERSION,
            true
        );

        wp_localize_script( 'converge-admin', 'convergeAdmin', array(
            'apiBase'      => CONVERGE_API_BASE,
            'ajaxUrl'      => admin_url( 'admin-ajax.php' ),
            'nonce'        => wp_create_nonce( 'converge_auth_nonce' ),
            'syncNonce'    => wp_create_nonce( 'converge_sync_nonce' ),
            'companyNonce' => wp_create_nonce( 'converge_company_nonce' ),
            'jobNonce'     => wp_create_nonce( 'converge_job_nonce' ),
            'token'        => $this->auth->get_access_token() ?: '',
            'connected'    => $this->auth->is_connected(),
        ) );
    }

    /**
     * Show admin notice if not connected.
     */
    public function show_connection_notice() {
        $screen = get_current_screen();
        if ( ! $screen || strpos( $screen->id, 'converge' ) === false ) {
            return;
        }

        // Check if refresh token expired and user needs to reconnect
        if ( get_transient( 'converge_reconnect_needed' ) ) {
            echo '<div class="notice notice-error"><p>';
            printf(
                /* translators: %s: settings page URL */
                esc_html__( 'Your Converge session has expired. %sReconnect your account%s to restore access.', 'converge-applybyvoice' ),
                '<a href="' . esc_url( admin_url( 'admin.php?page=converge-settings' ) ) . '">',
                '</a>'
            );
            echo '</p></div>';
        } elseif ( ! $this->auth->is_connected() && $screen->id !== 'converge_page_converge-settings' ) {
            echo '<div class="notice notice-warning"><p>';
            printf(
                /* translators: %s: settings page URL */
                esc_html__( 'Converge is not connected. %sConnect your account%s to get started.', 'converge-applybyvoice' ),
                '<a href="' . esc_url( admin_url( 'admin.php?page=converge-settings' ) ) . '">',
                '</a>'
            );
            echo '</p></div>';
        }
    }

    // --- Render Methods ---

    public function render_dashboard() {
        include CONVERGE_PLUGIN_DIR . 'templates/dashboard.php';
    }

    public function render_jobs() {
        include CONVERGE_PLUGIN_DIR . 'templates/jobs.php';
    }

    public function render_analytics() {
        include CONVERGE_PLUGIN_DIR . 'templates/analytics.php';
    }

    public function render_company() {
        include CONVERGE_PLUGIN_DIR . 'templates/company.php';
    }

    public function render_job_detail() {
        include CONVERGE_PLUGIN_DIR . 'templates/job-detail.php';
    }

    public function render_conversation() {
        include CONVERGE_PLUGIN_DIR . 'templates/conversation-detail.php';
    }

    public function render_settings() {
        include CONVERGE_PLUGIN_DIR . 'templates/settings.php';
    }

    // --- AJAX Handlers ---

    /**
     * AJAX: Save a plugin setting.
     */
    public function ajax_save_setting() {
        check_ajax_referer( 'converge_settings_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $key   = sanitize_key( wp_unslash( $_POST['key'] ?? '' ) );
        $value = sanitize_text_field( wp_unslash( $_POST['value'] ?? '' ) );

        $allowed_keys = array( 'converge_sync_enabled', 'converge_sync_frequency', 'converge_widget_mode' );
        if ( ! in_array( $key, $allowed_keys, true ) ) {
            wp_send_json_error( 'Invalid setting key' );
        }

        update_option( $key, $value );

        // Reschedule sync cron if frequency changed
        if ( $key === 'converge_sync_frequency' || $key === 'converge_sync_enabled' ) {
            $cron = new Converge_Cron( $this->auth, $this->sync );
            $cron->schedule_sync();
        }

        wp_send_json_success();
    }

    /**
     * AJAX: Save company info to Converge API.
     */
    public function ajax_save_company_info() {
        check_ajax_referer( 'converge_company_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        // Overview fields
        $fields = array(
            'company_description'          => sanitize_textarea_field( wp_unslash( $_POST['company_description'] ?? '' ) ),
            'company_mission'              => sanitize_textarea_field( wp_unslash( $_POST['company_mission'] ?? '' ) ),
            'company_culture'              => sanitize_textarea_field( wp_unslash( $_POST['company_culture'] ?? '' ) ),
            'company_size'                 => sanitize_text_field( wp_unslash( $_POST['company_size'] ?? '' ) ),
            'remote_work_policy'           => sanitize_text_field( wp_unslash( $_POST['remote_work_policy'] ?? '' ) ),
            'career_growth_opportunities'  => sanitize_textarea_field( wp_unslash( $_POST['career_growth_opportunities'] ?? '' ) ),
            'custom_greeting'              => sanitize_textarea_field( wp_unslash( $_POST['custom_greeting'] ?? '' ) ),
            'use_custom_greeting'          => ! empty( $_POST['use_custom_greeting'] ),
        );

        // Parse comma-separated values into JSON array
        $values_raw = sanitize_text_field( wp_unslash( $_POST['company_values'] ?? '' ) );
        if ( ! empty( $values_raw ) ) {
            $values_arr = array_map( 'trim', explode( ',', $values_raw ) );
            $fields['company_values'] = array_map( function( $v ) {
                return array( 'name' => $v, 'description' => '' );
            }, array_filter( $values_arr ) );
        } else {
            $fields['company_values'] = array();
        }

        // Benefits (stored as JSONB)
        $benefits = array();
        $benefit_keys = array( 'health', 'dental', 'vision', 'retirement', 'equity', 'bonus' );
        foreach ( $benefit_keys as $bk ) {
            $val = sanitize_textarea_field( wp_unslash( $_POST[ 'benefit_' . $bk ] ?? '' ) );
            if ( ! empty( $val ) ) {
                $benefits[ $bk ] = $val;
            }
        }
        $fields['benefits'] = $benefits;

        $fields['pto_policy']        = sanitize_text_field( wp_unslash( $_POST['pto_policy'] ?? '' ) );
        $fields['vacation_days']     = intval( $_POST['vacation_days'] ?? 0 ) ?: null;
        $fields['sick_days']         = intval( $_POST['sick_days'] ?? 0 ) ?: null;
        $fields['parental_leave']    = sanitize_textarea_field( wp_unslash( $_POST['parental_leave'] ?? '' ) );
        $fields['learning_budget']   = sanitize_text_field( wp_unslash( $_POST['learning_budget'] ?? '' ) );

        // Work details
        $fields['remote_work_details']    = sanitize_textarea_field( wp_unslash( $_POST['remote_work_details'] ?? '' ) );
        $fields['work_hours_flexibility'] = sanitize_text_field( wp_unslash( $_POST['work_hours_flexibility'] ?? '' ) );

        // Office locations (JSON array)
        $locations_json = wp_unslash( $_POST['office_locations'] ?? '[]' );
        $locations = json_decode( $locations_json, true );
        if ( is_array( $locations ) ) {
            $clean_locations = array();
            foreach ( $locations as $loc ) {
                $clean_locations[] = array(
                    'name'    => sanitize_text_field( $loc['name'] ?? '' ),
                    'address' => sanitize_text_field( $loc['address'] ?? '' ),
                    'is_hq'   => ! empty( $loc['is_hq'] ),
                );
            }
            $fields['office_locations'] = $clean_locations;
        }

        // Hiring
        $fields['interview_process_overview'] = sanitize_textarea_field( wp_unslash( $_POST['interview_process_overview'] ?? '' ) );
        $fields['typical_hiring_timeline']    = sanitize_text_field( wp_unslash( $_POST['typical_hiring_timeline'] ?? '' ) );

        $result = $this->api->update_company_info( $fields );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success();
    }

    /**
     * AJAX: Update job fields via Converge API.
     */
    public function ajax_update_job() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $job_id  = sanitize_text_field( wp_unslash( $_POST['job_id'] ?? '' ) );
        $section = sanitize_key( wp_unslash( $_POST['section'] ?? '' ) );

        if ( empty( $job_id ) || empty( $section ) ) {
            wp_send_json_error( 'Missing job_id or section' );
        }

        $data = array(
            'updateType' => 'job',
            'jobId'      => $job_id,
        );

        switch ( $section ) {
            case 'basic':
                $data['title']           = sanitize_text_field( wp_unslash( $_POST['title'] ?? '' ) );
                $data['location']        = sanitize_text_field( wp_unslash( $_POST['location'] ?? '' ) );
                $data['department']      = sanitize_text_field( wp_unslash( $_POST['department'] ?? '' ) );
                $data['employment_type'] = sanitize_text_field( wp_unslash( $_POST['employment_type'] ?? '' ) );
                $data['status']          = sanitize_text_field( wp_unslash( $_POST['status'] ?? '' ) );
                $data['experience_level'] = sanitize_text_field( wp_unslash( $_POST['experience_level'] ?? '' ) );
                break;

            case 'knowledge':
                $data['job_knowledge_base'] = array(
                    'salary_range_min'    => sanitize_text_field( wp_unslash( $_POST['salary_range_min'] ?? '' ) ),
                    'salary_range_max'    => sanitize_text_field( wp_unslash( $_POST['salary_range_max'] ?? '' ) ),
                    'salary_currency'     => sanitize_text_field( wp_unslash( $_POST['salary_currency'] ?? 'USD' ) ),
                    'salary_negotiable'   => ! empty( $_POST['salary_negotiable'] ),
                    'bonus_structure'     => sanitize_textarea_field( wp_unslash( $_POST['bonus_structure'] ?? '' ) ),
                    'team_size'           => sanitize_text_field( wp_unslash( $_POST['team_size'] ?? '' ) ),
                    'direct_reports'      => sanitize_text_field( wp_unslash( $_POST['direct_reports'] ?? '' ) ),
                    'reporting_to'        => sanitize_text_field( wp_unslash( $_POST['reporting_to'] ?? '' ) ),
                    'team_description'    => sanitize_textarea_field( wp_unslash( $_POST['team_description'] ?? '' ) ),
                    'next_role'           => sanitize_text_field( wp_unslash( $_POST['next_role'] ?? '' ) ),
                    'promotion_timeline'  => sanitize_text_field( wp_unslash( $_POST['promotion_timeline'] ?? '' ) ),
                    'skill_development'   => sanitize_textarea_field( wp_unslash( $_POST['skill_development'] ?? '' ) ),
                    'day_in_life'         => sanitize_textarea_field( wp_unslash( $_POST['day_in_life'] ?? '' ) ),
                    'travel_requirements' => sanitize_text_field( wp_unslash( $_POST['travel_requirements'] ?? '' ) ),
                    'sponsorship_available' => ! empty( $_POST['sponsorship_available'] ),
                );
                break;

            case 'screening':
                $collect_fields = array(
                    'collect_salary_expectations', 'collect_availability',
                    'collect_location', 'collect_notice_period', 'collect_employment_status',
                    'collect_work_authorization', 'collect_relocation',
                );
                $screening = array();
                foreach ( $collect_fields as $field ) {
                    $screening[ $field ] = ! empty( $_POST[ $field ] );
                }

                // Knockout requirements
                $requirement_keys = array(
                    'work_authorization', 'minimum_age', 'background_check', 'drug_screening',
                    'work_weekends', 'work_evenings', 'willing_to_travel', 'willing_to_relocate',
                    'drivers_license', 'reliable_transportation',
                );
                $requirements = array();
                foreach ( $requirement_keys as $key ) {
                    $enabled = ! empty( $_POST[ 'req_' . $key ] );
                    $behavior = sanitize_text_field( wp_unslash( $_POST[ 'req_behavior_' . $key ] ?? 'flag' ) );
                    if ( ! in_array( $behavior, array( 'flag', 'end' ), true ) ) {
                        $behavior = 'flag';
                    }
                    $requirements[ $key ] = array(
                        'enabled'  => $enabled,
                        'behavior' => $behavior,
                    );
                }
                $screening['requirements'] = $requirements;

                // Custom requirements
                $custom_reqs_json = wp_unslash( $_POST['custom_requirements'] ?? '[]' );
                $custom_reqs = json_decode( $custom_reqs_json, true );
                if ( is_array( $custom_reqs ) ) {
                    $clean_custom = array();
                    foreach ( $custom_reqs as $cr ) {
                        $q = sanitize_text_field( $cr['question'] ?? '' );
                        $d = sanitize_text_field( $cr['disqualify_on'] ?? 'no' );
                        if ( ! empty( $q ) ) {
                            $clean_custom[] = array( 'question' => $q, 'disqualify_on' => $d );
                        }
                    }
                    $screening['custom_requirements'] = $clean_custom;
                }

                $data['screening_requirements'] = $screening;
                break;

            case 'ai':
                $ai = array();
                $ai['personality'] = sanitize_text_field( wp_unslash( $_POST['personality'] ?? 'professional' ) );
                $ai['pace']        = sanitize_text_field( wp_unslash( $_POST['pace'] ?? 'moderate' ) );
                $ai['probing_depth'] = intval( $_POST['probing_depth'] ?? 3 );
                if ( $ai['probing_depth'] < 1 ) $ai['probing_depth'] = 1;
                if ( $ai['probing_depth'] > 5 ) $ai['probing_depth'] = 5;

                $voice = sanitize_text_field( wp_unslash( $_POST['voice'] ?? '' ) );
                $voice_name = sanitize_text_field( wp_unslash( $_POST['voice_name'] ?? '' ) );
                if ( ! empty( $voice ) ) {
                    $ai['voice']      = $voice;
                    $ai['voice_name'] = $voice_name;
                }

                $ai['language'] = sanitize_text_field( wp_unslash( $_POST['language'] ?? 'en' ) );

                // Allowed languages from checkboxes
                $allowed = array();
                if ( ! empty( $_POST['allowed_languages'] ) && is_array( $_POST['allowed_languages'] ) ) {
                    foreach ( $_POST['allowed_languages'] as $lang ) {
                        $allowed[] = sanitize_text_field( wp_unslash( $lang ) );
                    }
                }
                if ( empty( $allowed ) ) {
                    $allowed = array( 'en' );
                }
                $ai['allowed_languages'] = $allowed;

                $ai['prosody_enabled'] = ! empty( $_POST['prosody_enabled'] );
                $ai['custom_instructions'] = sanitize_textarea_field( wp_unslash( $_POST['custom_instructions'] ?? '' ) );

                $data['ai_conversation_style'] = $ai;
                break;

            case 'competencies':
                $competencies_json = wp_unslash( $_POST['competencies'] ?? '[]' );
                $competencies = json_decode( $competencies_json, true );
                if ( ! is_array( $competencies ) ) {
                    $competencies = array();
                }

                $clean = array();
                foreach ( $competencies as $comp ) {
                    $clean[] = array(
                        'name'            => sanitize_text_field( $comp['name'] ?? '' ),
                        'weight'          => floatval( $comp['weight'] ?? 0 ),
                        'signals'         => array_map( 'sanitize_text_field', (array) ( $comp['signals'] ?? array() ) ),
                        'sampleQuestions' => array_map( 'sanitize_text_field', (array) ( $comp['sampleQuestions'] ?? array() ) ),
                    );
                }
                $data['parsed_competencies'] = $clean;
                break;

            case 'conversation':
                $config_json = wp_unslash( $_POST['conversation_config'] ?? '{}' );
                $config = json_decode( $config_json, true );
                if ( ! is_array( $config ) ) {
                    $config = array();
                }
                $data['conversation_config'] = $config;
                break;

            default:
                wp_send_json_error( 'Invalid section' );
                return;
        }

        $result = $this->api->patch( '/api/auth/wp-token', $data );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success();
    }

    /**
     * AJAX: Parse competencies from job description.
     */
    public function ajax_parse_competencies() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $job_id      = sanitize_text_field( wp_unslash( $_POST['job_id'] ?? '' ) );
        $description = wp_unslash( $_POST['description'] ?? '' );

        if ( empty( $job_id ) || empty( $description ) ) {
            wp_send_json_error( 'Missing job_id or description' );
        }

        $result = $this->api->post( '/api/auth/wp-token', array(
            'action'      => 'parseCompetencies',
            'description' => $description,
            'jobId'       => $job_id,
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Generate interview questions for a competency.
     */
    public function ajax_generate_questions() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $competency_name = sanitize_text_field( wp_unslash( $_POST['competency_name'] ?? '' ) );
        $job_title       = sanitize_text_field( wp_unslash( $_POST['job_title'] ?? '' ) );
        $language        = sanitize_text_field( wp_unslash( $_POST['language'] ?? 'en' ) );

        if ( empty( $competency_name ) ) {
            wp_send_json_error( 'Missing competency name' );
        }

        $signals_raw = wp_unslash( $_POST['signals'] ?? '' );
        $signals     = array_filter( array_map( 'trim', explode( "\n", $signals_raw ) ) );

        $result = $this->api->post( '/api/auth/wp-token', array(
            'action'         => 'generateQuestions',
            'competencyName' => $competency_name,
            'signals'        => array_values( $signals ),
            'jobTitle'       => $job_title,
            'language'       => $language,
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Update review status for a conversation.
     */
    public function ajax_update_review() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $conversation_id = sanitize_text_field( wp_unslash( $_POST['conversation_id'] ?? '' ) );
        $review_status   = sanitize_text_field( wp_unslash( $_POST['review_status'] ?? '' ) );

        if ( empty( $conversation_id ) || empty( $review_status ) ) {
            wp_send_json_error( 'Missing conversation_id or review_status' );
        }

        $result = $this->api->post( '/api/auth/wp-token', array(
            'action'          => 'updateReviewStatus',
            'conversationId'  => $conversation_id,
            'reviewStatus'    => $review_status,
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Check insights status for polling.
     */
    public function ajax_check_insights() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $conversation_id = sanitize_text_field( wp_unslash( $_GET['conversation_id'] ?? $_POST['conversation_id'] ?? '' ) );

        if ( empty( $conversation_id ) ) {
            wp_send_json_error( 'Missing conversation_id' );
        }

        $result = $this->api->get( '/api/auth/wp-token', array(
            'data'           => 'insightsStatus',
            'conversationId' => $conversation_id,
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Export analytics as CSV.
     */
    public function ajax_export_analytics() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $result = $this->api->get( '/api/auth/wp-token', array(
            'data' => 'analyticsExport',
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Invite team member.
     */
    public function ajax_invite_member() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $email = sanitize_email( wp_unslash( $_POST['email'] ?? '' ) );
        $role  = sanitize_text_field( wp_unslash( $_POST['role'] ?? 'member' ) );

        if ( empty( $email ) ) {
            wp_send_json_error( 'Missing email' );
        }

        $result = $this->api->post( '/api/auth/wp-token', array(
            'action' => 'inviteMember',
            'email'  => $email,
            'role'   => $role,
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Remove team member.
     */
    public function ajax_remove_member() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $user_id = sanitize_text_field( wp_unslash( $_POST['user_id'] ?? '' ) );

        if ( empty( $user_id ) ) {
            wp_send_json_error( 'Missing user_id' );
        }

        $result = $this->api->post( '/api/auth/wp-token', array(
            'action' => 'removeMember',
            'userId' => $user_id,
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Create Stripe checkout session.
     */
    public function ajax_create_checkout() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $result = $this->api->post( '/api/auth/wp-token', array(
            'action' => 'createCheckout',
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Create Stripe billing portal session.
     */
    public function ajax_billing_portal() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $result = $this->api->post( '/api/auth/wp-token', array(
            'action' => 'billingPortal',
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Get team members.
     */
    public function ajax_get_team_members() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $result = $this->api->get( '/api/auth/wp-token', array( 'data' => 'teamMembers' ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Get AI configuration.
     */
    public function ajax_get_ai_config() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $result = $this->api->get( '/api/auth/wp-token', array( 'data' => 'aiConfig' ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Save AI configuration defaults.
     */
    public function ajax_save_ai_config() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $config = array();
        if ( isset( $_POST['default_personality'] ) ) {
            $config['default_personality'] = sanitize_text_field( wp_unslash( $_POST['default_personality'] ) );
        }
        if ( isset( $_POST['default_pace'] ) ) {
            $config['default_pace'] = sanitize_text_field( wp_unslash( $_POST['default_pace'] ) );
        }
        if ( isset( $_POST['default_probing_depth'] ) ) {
            $config['default_probing_depth'] = intval( $_POST['default_probing_depth'] );
        }
        if ( isset( $_POST['prosody_analysis_enabled'] ) ) {
            $config['prosody_analysis_enabled'] = ( $_POST['prosody_analysis_enabled'] === 'true' || $_POST['prosody_analysis_enabled'] === '1' );
        }
        if ( isset( $_POST['hume_voice_id'] ) ) {
            $config['hume_voice_id'] = sanitize_text_field( wp_unslash( $_POST['hume_voice_id'] ) );
        }
        if ( isset( $_POST['hume_voice_name'] ) ) {
            $config['hume_voice_name'] = sanitize_text_field( wp_unslash( $_POST['hume_voice_name'] ) );
        }
        if ( isset( $_POST['screening_defaults'] ) ) {
            $screening = json_decode( wp_unslash( $_POST['screening_defaults'] ), true );
            if ( is_array( $screening ) ) {
                $config['screeningDefaults'] = $screening;
            }
        }
        if ( isset( $_POST['conversation_config'] ) ) {
            $convConfig = json_decode( wp_unslash( $_POST['conversation_config'] ), true );
            if ( is_array( $convConfig ) ) {
                $config['conversationConfig'] = $convConfig;
            }
        }
        if ( isset( $_POST['custom_greeting'] ) ) {
            $config['customGreeting'] = sanitize_textarea_field( wp_unslash( $_POST['custom_greeting'] ) );
        }
        if ( isset( $_POST['use_custom_greeting'] ) ) {
            $config['useCustomGreeting'] = ( $_POST['use_custom_greeting'] === 'true' || $_POST['use_custom_greeting'] === '1' );
        }

        $result = $this->api->post( '/api/auth/wp-token', array(
            'action' => 'updateAIConfig',
            'config' => $config,
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Save organization settings (automation, data retention, portal).
     */
    public function ajax_save_org_settings() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $settings = array();
        if ( isset( $_POST['widget_auto_create'] ) ) {
            $settings['widget_auto_create'] = ( $_POST['widget_auto_create'] === 'true' || $_POST['widget_auto_create'] === '1' );
        }
        if ( isset( $_POST['widget_auto_publish'] ) ) {
            $settings['widget_auto_publish'] = ( $_POST['widget_auto_publish'] === 'true' || $_POST['widget_auto_publish'] === '1' );
        }
        if ( isset( $_POST['data_retention_days'] ) ) {
            $settings['data_retention_days'] = intval( $_POST['data_retention_days'] );
        }
        if ( isset( $_POST['candidate_portal_settings'] ) ) {
            $portal = json_decode( wp_unslash( $_POST['candidate_portal_settings'] ), true );
            if ( is_array( $portal ) ) {
                $settings['candidate_portal_settings'] = $portal;
            }
        }

        $result = $this->api->post( '/api/auth/wp-token', array(
            'action'   => 'updateOrgSettings',
            'settings' => $settings,
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }

    /**
     * AJAX: Get enhanced analytics data for charts.
     */
    public function ajax_get_enhanced_analytics() {
        check_ajax_referer( 'converge_job_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $days = isset( $_GET['days'] ) ? intval( $_GET['days'] ) : 30;

        $result = $this->api->get( '/api/auth/wp-token', array(
            'data' => 'enhancedAnalytics',
            'days' => $days,
        ) );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        wp_send_json_success( $result );
    }
}
