<?php
/**
 * API client for Converge WordPress plugin.
 * Handles all communication with the Converge API.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Converge_API {

    private $auth;

    public function __construct( Converge_Auth $auth ) {
        $this->auth = $auth;
    }

    /**
     * Make an authenticated GET request to the Converge API.
     *
     * @param string $endpoint API endpoint path.
     * @param array  $params   Optional query parameters to append.
     */
    public function get( $endpoint, $params = array() ) {
        if ( ! empty( $params ) ) {
            $endpoint .= ( strpos( $endpoint, '?' ) !== false ? '&' : '?' ) . http_build_query( $params );
        }
        return $this->request( 'GET', $endpoint );
    }

    /**
     * Make an authenticated POST request to the Converge API.
     */
    public function post( $endpoint, $data = array() ) {
        return $this->request( 'POST', $endpoint, $data );
    }

    /**
     * Refresh the access token using the refresh token.
     */
    public function refresh_token() {
        $refresh_token = $this->auth->get_refresh_token();
        if ( ! $refresh_token ) {
            return new WP_Error( 'no_refresh_token', 'No refresh token available' );
        }

        $response = wp_remote_post( CONVERGE_API_BASE . '/api/auth/wp-token', array(
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body'    => wp_json_encode( array(
                'refreshToken' => $refresh_token,
            ) ),
            'timeout' => 15,
        ) );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( $code === 401 ) {
            // Refresh token expired — user must reconnect
            $this->auth->disconnect();
            return new WP_Error( 'refresh_expired', 'Session expired. Please reconnect.' );
        }

        if ( $code >= 400 || empty( $body['accessToken'] ) ) {
            return new WP_Error( 'refresh_failed', 'Token refresh failed' );
        }

        $this->auth->update_access_token( $body['accessToken'], $body['expiresAt'] ?? null );

        if ( ! empty( $body['refreshToken'] ) ) {
            $this->auth->update_refresh_token( $body['refreshToken'] );
        }

        return true;
    }

    /**
     * Make an authenticated PATCH request to the Converge API.
     */
    public function patch( $endpoint, $data = array() ) {
        return $this->request( 'PATCH', $endpoint, $data );
    }

    /**
     * Central request method with automatic 401 retry.
     * On 401, refreshes the access token and retries once.
     * If refresh token is expired, sets a transient so the admin notice can alert the user.
     */
    private function request( $method, $endpoint, $data = null, $is_retry = false ) {
        $token = $this->auth->get_access_token();
        if ( ! $token ) {
            return new WP_Error( 'not_connected', 'Not connected to Converge' );
        }

        $args = array(
            'method'  => $method,
            'headers' => array(
                'Authorization' => 'Bearer ' . $token,
                'Content-Type'  => 'application/json',
            ),
            'timeout' => $method === 'POST' ? 60 : 30,
        );

        if ( $data !== null ) {
            $args['body'] = wp_json_encode( $data );
        }

        $response = wp_remote_request( CONVERGE_API_BASE . $endpoint, $args );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( $code === 401 && ! $is_retry ) {
            // Token expired — attempt automatic refresh and retry
            $refresh_result = $this->refresh_token();
            if ( is_wp_error( $refresh_result ) ) {
                if ( $refresh_result->get_error_code() === 'refresh_expired' ) {
                    set_transient( 'converge_reconnect_needed', true, DAY_IN_SECONDS );
                }
                return $refresh_result;
            }
            // Retry with new token
            return $this->request( $method, $endpoint, $data, true );
        }

        if ( $code === 401 ) {
            return new WP_Error( 'unauthorized', 'Token expired or invalid' );
        }

        if ( $code >= 400 ) {
            return new WP_Error( 'api_error', $body['error'] ?? 'API request failed' );
        }

        return $body;
    }

    /**
     * Get org summary for WP dashboard widget.
     */
    public function get_org_summary() {
        return $this->get( '/api/auth/wp-token' );
    }

    /**
     * Get jobs list for WP Jobs page.
     */
    public function get_jobs() {
        return $this->get( '/api/auth/wp-token?data=jobs' );
    }

    /**
     * Get single job detail with candidates.
     */
    public function get_job_detail( $job_id ) {
        return $this->get( '/api/auth/wp-token?data=job&jobId=' . urlencode( $job_id ) );
    }

    /**
     * Get single conversation detail with insights, screening, and transcript.
     */
    public function get_conversation_detail( $conversation_id ) {
        return $this->get( '/api/auth/wp-token?data=conversation&conversationId=' . urlencode( $conversation_id ) );
    }

    /**
     * Get analytics for WP Analytics page.
     */
    public function get_analytics( $days = 30 ) {
        return $this->get( '/api/auth/wp-token?data=analytics&days=' . intval( $days ) );
    }

    /**
     * Get company info for WP Company page.
     */
    public function get_company_info() {
        return $this->get( '/api/auth/wp-token?data=company' );
    }

    /**
     * Update company info from WP Company page.
     */
    public function update_company_info( $data ) {
        return $this->patch( '/api/auth/wp-token', $data );
    }

    /**
     * Create or update a job via auto-match API.
     */
    public function sync_job( $job_data ) {
        return $this->post( '/api/widget/auto-match', $job_data );
    }
}
