<?php
/**
 * Authentication handler for Converge WordPress plugin.
 * Manages token storage, encryption, and refresh flow.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Converge_Auth {

    private $encryption_key;

    public function __construct() {
        $this->encryption_key = $this->get_encryption_key();

        add_action( 'wp_ajax_converge_save_auth', array( $this, 'ajax_save_auth' ) );
        add_action( 'wp_ajax_converge_disconnect', array( $this, 'ajax_disconnect' ) );
    }

    /**
     * Get encryption key derived from WordPress auth salt.
     */
    private function get_encryption_key() {
        return hash( 'sha256', wp_salt( 'auth' ) . 'converge_job_screener', true );
    }

    /**
     * Encrypt a value for storage.
     */
    private function encrypt( $value ) {
        if ( ! function_exists( 'openssl_encrypt' ) ) {
            return base64_encode( $value ); // Fallback if openssl not available
        }
        $iv = openssl_random_pseudo_bytes( 16 );
        $encrypted = openssl_encrypt( $value, 'aes-256-cbc', $this->encryption_key, 0, $iv );
        return base64_encode( $iv . '::' . $encrypted );
    }

    /**
     * Decrypt a stored value.
     */
    private function decrypt( $value ) {
        if ( ! function_exists( 'openssl_decrypt' ) ) {
            return base64_decode( $value );
        }
        $decoded = base64_decode( $value );
        $parts = explode( '::', $decoded, 2 );
        if ( count( $parts ) !== 2 ) {
            return false;
        }
        list( $iv, $encrypted ) = $parts;
        return openssl_decrypt( $encrypted, 'aes-256-cbc', $this->encryption_key, 0, $iv );
    }

    /**
     * Check if plugin is connected to a Converge account.
     */
    public function is_connected() {
        return ! empty( get_option( 'converge_access_token' ) );
    }

    /**
     * Get the current access token (decrypted).
     */
    public function get_access_token() {
        $encrypted = get_option( 'converge_access_token' );
        if ( empty( $encrypted ) ) {
            return false;
        }
        return $this->decrypt( $encrypted );
    }

    /**
     * Get the current refresh token (decrypted).
     */
    public function get_refresh_token() {
        $encrypted = get_option( 'converge_refresh_token' );
        if ( empty( $encrypted ) ) {
            return false;
        }
        return $this->decrypt( $encrypted );
    }

    /**
     * Store auth tokens from the popup flow.
     */
    public function save_tokens( $access_token, $refresh_token ) {
        update_option( 'converge_access_token', $this->encrypt( $access_token ) );
        update_option( 'converge_refresh_token', $this->encrypt( $refresh_token ) );
        update_option( 'converge_token_expires_at', time() + 3600 ); // ~1 hour
        update_option( 'converge_connected_at', time() );
        delete_transient( 'converge_reconnect_needed' );
    }

    /**
     * Update just the access token (after refresh).
     */
    public function update_access_token( $access_token, $expires_at = null ) {
        update_option( 'converge_access_token', $this->encrypt( $access_token ) );
        if ( $expires_at ) {
            update_option( 'converge_token_expires_at', $expires_at );
        }
    }

    /**
     * Update the refresh token (returned with each refresh).
     */
    public function update_refresh_token( $refresh_token ) {
        update_option( 'converge_refresh_token', $this->encrypt( $refresh_token ) );
    }

    /**
     * Clear all stored auth data.
     */
    public function disconnect() {
        delete_option( 'converge_access_token' );
        delete_option( 'converge_refresh_token' );
        delete_option( 'converge_token_expires_at' );
        delete_option( 'converge_connected_at' );
        delete_option( 'converge_org_name' );
        delete_option( 'converge_org_id' );
        delete_option( 'converge_org_slug' );
    }

    /**
     * Get connection info for display.
     */
    public function get_connection_info() {
        return array(
            'connected'    => $this->is_connected(),
            'org_name'     => get_option( 'converge_org_name', '' ),
            'org_id'       => get_option( 'converge_org_id', '' ),
            'connected_at' => get_option( 'converge_connected_at', '' ),
        );
    }

    /**
     * AJAX: Save auth tokens from popup.
     */
    public function ajax_save_auth() {
        check_ajax_referer( 'converge_auth_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $access_token  = sanitize_text_field( wp_unslash( $_POST['access_token'] ?? '' ) );
        $refresh_token = sanitize_text_field( wp_unslash( $_POST['refresh_token'] ?? '' ) );

        if ( empty( $access_token ) || empty( $refresh_token ) ) {
            wp_send_json_error( 'Missing tokens' );
        }

        $this->save_tokens( $access_token, $refresh_token );

        // Fetch org info to store display name
        $api = new Converge_API( $this );
        $org_data = $api->get_org_summary();

        if ( $org_data && ! is_wp_error( $org_data ) ) {
            update_option( 'converge_org_name', sanitize_text_field( $org_data['orgName'] ?? '' ) );
            update_option( 'converge_org_id', sanitize_text_field( $org_data['orgId'] ?? '' ) );
            update_option( 'converge_org_slug', sanitize_text_field( $org_data['orgSlug'] ?? '' ) );
        }

        wp_send_json_success( array(
            'org_name' => get_option( 'converge_org_name', 'Connected' ),
        ) );
    }

    /**
     * AJAX: Disconnect from Converge.
     */
    public function ajax_disconnect() {
        check_ajax_referer( 'converge_auth_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $this->disconnect();
        wp_send_json_success();
    }
}
