<?php
/**
 * WP-Cron scheduled tasks for Converge WordPress plugin.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Converge_Cron {

    private $auth;
    private $sync;

    public function __construct( Converge_Auth $auth, Converge_Sync $sync ) {
        $this->auth = $auth;
        $this->sync = $sync;

        add_action( 'converge_token_refresh', array( $this, 'refresh_token' ) );
        add_action( 'converge_job_sync', array( $this, 'sync_jobs' ) );
        add_filter( 'cron_schedules', array( $this, 'add_cron_schedules' ) );
    }

    /**
     * Add custom cron schedules.
     */
    public function add_cron_schedules( $schedules ) {
        $schedules['every_six_hours'] = array(
            'interval' => 6 * HOUR_IN_SECONDS,
            'display'  => __( 'Every 6 Hours', 'converge-applybyvoice' ),
        );
        $schedules['every_twelve_hours'] = array(
            'interval' => 12 * HOUR_IN_SECONDS,
            'display'  => __( 'Every 12 Hours', 'converge-applybyvoice' ),
        );
        return $schedules;
    }

    /**
     * Schedule cron events on plugin activation.
     */
    public function schedule_events() {
        if ( ! wp_next_scheduled( 'converge_token_refresh' ) ) {
            wp_schedule_event( time(), 'hourly', 'converge_token_refresh' );
        }

        $this->schedule_sync();
    }

    /**
     * Schedule the job sync event based on frequency setting.
     */
    public function schedule_sync() {
        // Clear existing sync schedule
        $timestamp = wp_next_scheduled( 'converge_job_sync' );
        if ( $timestamp ) {
            wp_unschedule_event( $timestamp, 'converge_job_sync' );
        }

        if ( get_option( 'converge_sync_enabled' ) !== '1' ) {
            return;
        }

        $frequency = get_option( 'converge_sync_frequency', 'daily' );

        // Map frequency to WP cron recurrence
        $recurrence_map = array(
            '6h'    => 'every_six_hours',
            '12h'   => 'every_twelve_hours',
            'daily' => 'daily',
            '48h'   => 'twicedaily', // Closest WP built-in
        );

        $recurrence = $recurrence_map[ $frequency ] ?? 'daily';
        wp_schedule_event( time(), $recurrence, 'converge_job_sync' );
    }

    /**
     * Clear all scheduled cron events.
     */
    public function clear_events() {
        $timestamp = wp_next_scheduled( 'converge_token_refresh' );
        if ( $timestamp ) {
            wp_unschedule_event( $timestamp, 'converge_token_refresh' );
        }

        $timestamp = wp_next_scheduled( 'converge_job_sync' );
        if ( $timestamp ) {
            wp_unschedule_event( $timestamp, 'converge_job_sync' );
        }
    }

    /**
     * Cron callback: Refresh the access token.
     */
    public function refresh_token() {
        if ( ! $this->auth->is_connected() ) {
            return;
        }

        $api = new Converge_API( $this->auth );
        $result = $api->refresh_token();

        if ( is_wp_error( $result ) ) {
            // Log the error
            error_log( 'Converge: Token refresh failed — ' . $result->get_error_message() );
        }
    }

    /**
     * Cron callback: Sync jobs from WP to Converge.
     */
    public function sync_jobs() {
        if ( ! $this->auth->is_connected() ) {
            return;
        }

        if ( get_option( 'converge_sync_enabled' ) !== '1' ) {
            return;
        }

        $results = $this->sync->sync_all();

        if ( isset( $results['error'] ) ) {
            error_log( 'Converge: Job sync failed — ' . $results['error'] );
        }
    }
}
