<?php
/**
 * Job board plugin detection and sync for Converge WordPress plugin.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Converge_Sync {

    private $api;

    /**
     * Supported job board plugins with their detection methods and metadata.
     */
    private $supported_plugins = array(
        'wp-job-manager' => array(
            'name'       => 'WP Job Manager',
            'cpt'        => 'job_listing',
            'detect'     => 'class_exists',
            'detect_arg' => 'WP_Job_Manager',
            'location'   => '_job_location',
            'company'    => '_company_name',
            'type'       => '_job_type',
        ),
        'simple-job-board' => array(
            'name'       => 'Simple Job Board',
            'cpt'        => 'jobpost',
            'detect'     => 'post_type_exists',
            'detect_arg' => 'jobpost',
            'location'   => 'jobpost_location', // taxonomy
            'company'    => '',
            'type'       => 'jobpost_job_type', // taxonomy
        ),
        'wp-job-openings' => array(
            'name'       => 'WP Job Openings',
            'cpt'        => 'awsm_job_openings',
            'detect'     => 'function_exists',
            'detect_arg' => 'awsm_jobs_init',
            'location'   => 'awsm_job_location', // taxonomy
            'company'    => '',
            'type'       => 'awsm_job_type', // taxonomy
        ),
    );

    public function __construct( Converge_API $api ) {
        $this->api = $api;

        add_action( 'wp_ajax_converge_sync_now', array( $this, 'ajax_sync_now' ) );

        // Register hooks for real-time sync on job changes
        add_action( 'save_post', array( $this, 'on_save_post' ), 20, 2 );
        add_action( 'trashed_post', array( $this, 'on_trash_post' ) );
    }

    /**
     * Detect which job board plugin is active.
     *
     * @return array|false Plugin info array or false if none detected.
     */
    public function detect_plugin() {
        foreach ( $this->supported_plugins as $slug => $plugin ) {
            $detected = false;

            switch ( $plugin['detect'] ) {
                case 'class_exists':
                    $detected = class_exists( $plugin['detect_arg'] );
                    break;
                case 'function_exists':
                    $detected = function_exists( $plugin['detect_arg'] );
                    break;
                case 'post_type_exists':
                    $detected = post_type_exists( $plugin['detect_arg'] );
                    break;
            }

            if ( $detected ) {
                return array_merge( $plugin, array( 'slug' => $slug ) );
            }
        }

        return false;
    }

    /**
     * Get all jobs from the detected WP job board plugin.
     *
     * @return array List of job data arrays.
     */
    public function get_wp_jobs() {
        $plugin = $this->detect_plugin();
        if ( ! $plugin ) {
            return array();
        }

        $posts = get_posts( array(
            'post_type'      => $plugin['cpt'],
            'post_status'    => 'publish',
            'posts_per_page' => 100,
            'orderby'        => 'date',
            'order'          => 'DESC',
        ) );

        $jobs = array();
        foreach ( $posts as $post ) {
            $jobs[] = $this->extract_job_data( $post, $plugin );
        }

        return $jobs;
    }

    /**
     * Extract standardized job data from a WP post.
     */
    private function extract_job_data( $post, $plugin ) {
        $data = array(
            'wp_post_id'  => $post->ID,
            'title'       => $post->post_title,
            'description' => $post->post_content,
            'url'         => get_permalink( $post->ID ),
            'location'    => '',
            'company'     => get_bloginfo( 'name' ),
            'type'        => '',
        );

        // Extract location
        if ( ! empty( $plugin['location'] ) ) {
            if ( taxonomy_exists( $plugin['location'] ) ) {
                $terms = wp_get_post_terms( $post->ID, $plugin['location'], array( 'fields' => 'names' ) );
                if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
                    $data['location'] = implode( ', ', $terms );
                }
            } else {
                $meta = get_post_meta( $post->ID, $plugin['location'], true );
                if ( $meta ) {
                    $data['location'] = $meta;
                }
            }
        }

        // Extract company name
        if ( ! empty( $plugin['company'] ) ) {
            $company = get_post_meta( $post->ID, $plugin['company'], true );
            if ( $company ) {
                $data['company'] = $company;
            }
        }

        // Extract employment type
        if ( ! empty( $plugin['type'] ) ) {
            if ( taxonomy_exists( $plugin['type'] ) ) {
                $terms = wp_get_post_terms( $post->ID, $plugin['type'], array( 'fields' => 'names' ) );
                if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
                    $data['type'] = $terms[0];
                }
            } else {
                $meta = get_post_meta( $post->ID, $plugin['type'], true );
                if ( $meta ) {
                    $data['type'] = $meta;
                }
            }
        }

        return $data;
    }

    /**
     * Sync all WP jobs to Converge.
     *
     * @return array Results with created, updated, failed counts.
     */
    public function sync_all() {
        if ( get_option( 'converge_sync_enabled' ) !== '1' ) {
            return array( 'error' => 'Sync is disabled' );
        }

        $jobs = $this->get_wp_jobs();
        $results = array(
            'total'   => count( $jobs ),
            'synced'  => 0,
            'failed'  => 0,
            'errors'  => array(),
        );

        foreach ( $jobs as $job ) {
            $result = $this->sync_single_job( $job );

            if ( is_wp_error( $result ) ) {
                $results['failed']++;
                $results['errors'][] = $job['title'] . ': ' . $result->get_error_message();
            } elseif ( empty( $result['matched'] ) ) {
                $results['failed']++;
                $results['errors'][] = $job['title'] . ': Job was not created in Converge';
            } else {
                $results['synced']++;

                // Store Converge job ID in post meta
                if ( ! empty( $result['jobId'] ) ) {
                    update_post_meta( $job['wp_post_id'], '_converge_job_id', $result['jobId'] );
                }
            }
        }

        // Update last sync timestamp
        update_option( 'converge_last_sync', time() );
        update_option( 'converge_last_sync_count', $results['synced'] );

        return $results;
    }

    /**
     * Sync a single job to Converge.
     */
    private function sync_single_job( $job ) {
        $org_slug = get_option( 'converge_org_slug', '' );
        $payload = array(
            'pageUrl'           => $job['url'],
            'extractedTitle'    => $job['title'],
            'extractedLocation' => $job['location'],
            'source'            => 'wordpress',
        );

        // Use slug if available, fall back to orgId for pre-slug installs
        if ( ! empty( $org_slug ) ) {
            $payload['orgSlug'] = $org_slug;
        } else {
            $payload['orgId'] = get_option( 'converge_org_id', '' );
        }

        // Include description as JSON-LD for auto-create
        $description = wp_strip_all_tags( $job['description'] );
        if ( ! empty( $description ) ) {
            $payload['jsonLd'] = array( 'description' => $description );
        }

        return $this->api->sync_job( $payload );
    }

    /**
     * Handle real-time sync when a job post is saved/published.
     */
    public function on_save_post( $post_id, $post ) {
        if ( get_option( 'converge_sync_enabled' ) !== '1' ) {
            return;
        }

        $plugin = $this->detect_plugin();
        if ( ! $plugin || $post->post_type !== $plugin['cpt'] ) {
            return;
        }

        if ( $post->post_status !== 'publish' ) {
            return;
        }

        // Don't sync during autosave
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        $job_data = $this->extract_job_data( $post, $plugin );
        $result = $this->sync_single_job( $job_data );

        if ( ! is_wp_error( $result ) && ! empty( $result['jobId'] ) ) {
            update_post_meta( $post_id, '_converge_job_id', $result['jobId'] );
        }
    }

    /**
     * Handle when a job post is trashed — archive in Converge.
     */
    public function on_trash_post( $post_id ) {
        if ( get_option( 'converge_sync_enabled' ) !== '1' ) {
            return;
        }

        $converge_job_id = get_post_meta( $post_id, '_converge_job_id', true );
        if ( ! $converge_job_id ) {
            return;
        }

        // Notify Converge to archive this job
        $this->api->post( '/api/jobs/' . $converge_job_id . '/archive', array(
            'source' => 'wordpress',
        ) );
    }

    /**
     * AJAX: Manual sync trigger.
     */
    public function ajax_sync_now() {
        check_ajax_referer( 'converge_sync_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }

        $results = $this->sync_all();
        wp_send_json_success( $results );
    }
}
