<?php
/**
 * Template: Native Conversation Detail page for Converge WordPress plugin.
 * Shows full candidate review — insights, screening data, and transcript.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

$auth = new Converge_Auth();
$connected = $auth->is_connected();
$conversation_id = isset( $_GET['conversation_id'] ) ? sanitize_text_field( wp_unslash( $_GET['conversation_id'] ) ) : '';
$job_id = isset( $_GET['job_id'] ) ? sanitize_text_field( wp_unslash( $_GET['job_id'] ) ) : '';
?>

<div class="wrap">
    <?php if ( ! $connected ) : ?>
        <h1><?php esc_html_e( 'Candidate Review', 'converge-applybyvoice' ); ?></h1>
        <div class="converge-card converge-connect-cta">
            <h2><?php esc_html_e( 'Connect Your Account', 'converge-applybyvoice' ); ?></h2>
            <p><?php esc_html_e( 'Connect your Converge account to review candidates.', 'converge-applybyvoice' ); ?></p>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=converge-settings' ) ); ?>" class="button button-primary">
                <?php esc_html_e( 'Connect Account', 'converge-applybyvoice' ); ?>
            </a>
        </div>
    <?php elseif ( empty( $conversation_id ) ) : ?>
        <h1><?php esc_html_e( 'Candidate Review', 'converge-applybyvoice' ); ?></h1>
        <div class="notice notice-error"><p><?php esc_html_e( 'No conversation specified.', 'converge-applybyvoice' ); ?></p></div>
    <?php else :
        $api = new Converge_API( $auth );
        $result = $api->get_conversation_detail( $conversation_id );

        if ( is_wp_error( $result ) ) : ?>
            <h1><?php esc_html_e( 'Candidate Review', 'converge-applybyvoice' ); ?></h1>
            <div class="notice notice-error"><p><?php echo esc_html( $result->get_error_message() ); ?></p></div>
        <?php else :
            $conv = $result['conversation'];
            $candidate = $result['candidate'];
            $job = $result['job'];
            $insights = $result['insights'];
            $screening = $result['screening'];
            $transcript = $result['transcript'] ?? array();
            $review_status = $conv['reviewStatus'] ?? 'new';

            $duration_str = '';
            if ( ! empty( $conv['duration'] ) ) {
                $duration_str = sprintf( '%d:%02d', floor( $conv['duration'] / 60 ), $conv['duration'] % 60 );
            }

            // Back link
            $back_url = ! empty( $job_id )
                ? admin_url( 'admin.php?page=converge-job-detail&job_id=' . urlencode( $job_id ) )
                : admin_url( 'admin.php?page=converge-jobs' );
        ?>

            <div class="converge-conversation-header">
                <div class="converge-conversation-header-left">
                    <a href="<?php echo esc_url( $back_url ); ?>" class="converge-back-link">
                        &larr; <?php esc_html_e( 'Back', 'converge-applybyvoice' ); ?>
                    </a>
                    <h1 class="wp-heading-inline">
                        <?php echo esc_html( $candidate['name'] ?? __( 'Unknown Candidate', 'converge-applybyvoice' ) ); ?>
                    </h1>
                    <span class="converge-review-badge converge-review-<?php echo esc_attr( $review_status ); ?>">
                        <?php echo esc_html( ucfirst( $review_status ) ); ?>
                    </span>
                </div>
                <div class="converge-review-actions" data-conversation-id="<?php echo esc_attr( $conversation_id ); ?>">
                    <?php if ( $review_status !== 'shortlisted' ) : ?>
                        <button type="button" class="button button-primary converge-review-action" data-status="shortlisted">
                            <span class="dashicons dashicons-star-filled" style="margin-top:3px;"></span>
                            <?php esc_html_e( 'Shortlist', 'converge-applybyvoice' ); ?>
                        </button>
                    <?php else : ?>
                        <button type="button" class="button converge-review-action" data-status="viewed">
                            <span class="dashicons dashicons-star-empty" style="margin-top:3px;"></span>
                            <?php esc_html_e( 'Remove from Shortlist', 'converge-applybyvoice' ); ?>
                        </button>
                    <?php endif; ?>
                    <?php if ( $review_status !== 'archived' ) : ?>
                        <button type="button" class="button converge-review-action" data-status="archived">
                            <span class="dashicons dashicons-archive" style="margin-top:3px;"></span>
                            <?php esc_html_e( 'Archive', 'converge-applybyvoice' ); ?>
                        </button>
                    <?php else : ?>
                        <button type="button" class="button converge-review-action" data-status="viewed">
                            <span class="dashicons dashicons-undo" style="margin-top:3px;"></span>
                            <?php esc_html_e( 'Unarchive', 'converge-applybyvoice' ); ?>
                        </button>
                    <?php endif; ?>
                </div>
            </div>
            <hr class="wp-header-end">

            <div class="converge-review-layout">
                <!-- Left Column: Main Content -->
                <div class="converge-review-main">

                    <?php if ( $insights ) : ?>

                        <!-- Summary -->
                        <?php if ( ! empty( $insights['summary'] ) ) : ?>
                            <div class="converge-card">
                                <h2><?php esc_html_e( 'Summary', 'converge-applybyvoice' ); ?></h2>
                                <p><?php echo esc_html( $insights['summary'] ); ?></p>
                            </div>
                        <?php endif; ?>

                        <!-- Key Strengths & Areas to Explore -->
                        <?php if ( ! empty( $insights['strengths'] ) || ! empty( $insights['areasToExplore'] ) ) : ?>
                            <div class="converge-review-columns">
                                <?php if ( ! empty( $insights['strengths'] ) ) : ?>
                                    <div class="converge-card">
                                        <h2><?php esc_html_e( 'Key Strengths', 'converge-applybyvoice' ); ?></h2>
                                        <ul class="converge-evidence-list converge-strengths">
                                            <?php foreach ( $insights['strengths'] as $s ) : ?>
                                                <li><?php echo esc_html( $s ); ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                                <?php if ( ! empty( $insights['areasToExplore'] ) ) : ?>
                                    <div class="converge-card">
                                        <h2><?php esc_html_e( 'Areas to Explore', 'converge-applybyvoice' ); ?></h2>
                                        <ul class="converge-evidence-list converge-explore">
                                            <?php foreach ( $insights['areasToExplore'] as $a ) : ?>
                                                <li><?php echo esc_html( $a ); ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>

                        <!-- Competency Assessment -->
                        <?php if ( ! empty( $insights['capabilityIndicators'] ) ) : ?>
                            <div class="converge-card">
                                <h2><?php esc_html_e( 'Competency Assessment', 'converge-applybyvoice' ); ?></h2>
                                <?php foreach ( $insights['capabilityIndicators'] as $ci ) : ?>
                                    <div class="converge-competency-item">
                                        <h3><?php echo esc_html( $ci['competency'] ); ?></h3>
                                        <?php if ( ! empty( $ci['evidence'] ) ) : ?>
                                            <p><strong><?php esc_html_e( 'Evidence:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $ci['evidence'] ); ?></p>
                                        <?php endif; ?>
                                        <?php if ( ! empty( $ci['exploreFurther'] ) ) : ?>
                                            <p class="converge-explore-further"><strong><?php esc_html_e( 'Explore further:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $ci['exploreFurther'] ); ?></p>
                                        <?php endif; ?>
                                        <?php if ( ! empty( $ci['engagementNote'] ) ) : ?>
                                            <p class="converge-engagement-note"><?php echo esc_html( $ci['engagementNote'] ); ?></p>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>

                                <?php if ( ! empty( $insights['potentialGaps'] ) ) : ?>
                                    <h3 style="margin-top: 16px;"><?php esc_html_e( 'Potential Gaps', 'converge-applybyvoice' ); ?></h3>
                                    <?php foreach ( $insights['potentialGaps'] as $gap ) : ?>
                                        <div class="converge-gap-item">
                                            <strong><?php echo esc_html( $gap['area'] ); ?>:</strong>
                                            <?php echo esc_html( $gap['reason'] ); ?>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>

                        <!-- Background, Motivation, Culture Fit -->
                        <?php
                        $bg = $insights['backgroundInsight'] ?? null;
                        $mo = $insights['motivationInsight'] ?? null;
                        $cf = $insights['cultureFitInsight'] ?? null;
                        ?>
                        <?php if ( $bg || $mo || $cf ) : ?>
                            <div class="converge-card">
                                <h2><?php esc_html_e( 'Candidate Profile', 'converge-applybyvoice' ); ?></h2>

                                <?php if ( $bg ) : ?>
                                    <div class="converge-profile-section">
                                        <h3><?php esc_html_e( 'Background', 'converge-applybyvoice' ); ?></h3>
                                        <?php if ( ! empty( $bg['career_summary'] ) ) : ?>
                                            <p><?php echo esc_html( $bg['career_summary'] ); ?></p>
                                        <?php endif; ?>
                                        <?php if ( ! empty( $bg['key_achievements'] ) && is_array( $bg['key_achievements'] ) ) : ?>
                                            <ul class="converge-insight-list">
                                                <?php foreach ( $bg['key_achievements'] as $ach ) : ?>
                                                    <li><?php echo esc_html( $ach ); ?></li>
                                                <?php endforeach; ?>
                                            </ul>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>

                                <?php if ( $mo ) : ?>
                                    <div class="converge-profile-section">
                                        <h3><?php esc_html_e( 'Motivation', 'converge-applybyvoice' ); ?></h3>
                                        <?php if ( ! empty( $mo['why_this_role'] ) ) : ?>
                                            <p><strong><?php esc_html_e( 'Why this role:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $mo['why_this_role'] ); ?></p>
                                        <?php endif; ?>
                                        <?php if ( ! empty( $mo['career_goals'] ) ) : ?>
                                            <p><strong><?php esc_html_e( 'Career goals:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $mo['career_goals'] ); ?></p>
                                        <?php endif; ?>
                                        <?php if ( ! empty( $mo['what_matters_most'] ) ) : ?>
                                            <p><strong><?php esc_html_e( 'What matters most:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $mo['what_matters_most'] ); ?></p>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>

                                <?php if ( $cf ) : ?>
                                    <div class="converge-profile-section">
                                        <h3><?php esc_html_e( 'Culture Fit', 'converge-applybyvoice' ); ?></h3>
                                        <?php if ( ! empty( $cf['work_environment_preference'] ) ) : ?>
                                            <p><strong><?php esc_html_e( 'Work environment:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $cf['work_environment_preference'] ); ?></p>
                                        <?php endif; ?>
                                        <?php if ( ! empty( $cf['management_style_preference'] ) ) : ?>
                                            <p><strong><?php esc_html_e( 'Management style:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $cf['management_style_preference'] ); ?></p>
                                        <?php endif; ?>
                                        <?php if ( ! empty( $cf['collaboration_style'] ) ) : ?>
                                            <p><strong><?php esc_html_e( 'Collaboration:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $cf['collaboration_style'] ); ?></p>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>

                        <!-- Voice Engagement -->
                        <?php if ( ! empty( $insights['prosodyObservations'] ) ) : ?>
                            <div class="converge-card">
                                <h2><?php esc_html_e( 'Voice Engagement', 'converge-applybyvoice' ); ?></h2>
                                <ul class="converge-evidence-list">
                                    <?php foreach ( $insights['prosodyObservations'] as $obs ) : ?>
                                        <li><?php echo esc_html( $obs ); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endif; ?>

                        <!-- Follow-up Suggestions with Priority Badges -->
                        <?php if ( ! empty( $insights['followUpSuggestions'] ) ) : ?>
                            <div class="converge-card">
                                <h2><?php esc_html_e( 'Suggested Follow-up Questions', 'converge-applybyvoice' ); ?></h2>
                                <div class="converge-followup-list">
                                    <?php foreach ( $insights['followUpSuggestions'] as $idx => $fus ) :
                                        $is_array = is_array( $fus );
                                        $text = $is_array ? ( $fus['question'] ?? $fus['text'] ?? '' ) : $fus;
                                        $priority = $is_array ? ( $fus['priority'] ?? '' ) : '';
                                        $rationale = $is_array ? ( $fus['rationale'] ?? '' ) : '';
                                        // Assign priority based on position if not provided
                                        if ( empty( $priority ) ) {
                                            $priority = $idx === 0 ? 'high' : ( $idx === 1 ? 'medium' : 'low' );
                                        }
                                    ?>
                                        <div class="converge-followup-item">
                                            <div class="converge-followup-header">
                                                <span class="converge-priority-badge converge-priority-<?php echo esc_attr( $priority ); ?>">
                                                    <?php echo esc_html( ucfirst( $priority ) ); ?>
                                                </span>
                                                <span class="converge-followup-text"><?php echo esc_html( $text ); ?></span>
                                            </div>
                                            <?php if ( ! empty( $rationale ) ) : ?>
                                                <p class="converge-followup-rationale"><?php echo esc_html( $rationale ); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endif; ?>

                    <?php elseif ( $conv['insightsStatus'] === 'pending' || $conv['insightsStatus'] === 'processing' ) : ?>
                        <div class="converge-card converge-analyzing-card" data-conversation-id="<?php echo esc_attr( $conversation_id ); ?>">
                            <div class="converge-analyzing-spinner">
                                <div class="converge-spinner-dots">
                                    <span></span><span></span><span></span>
                                </div>
                                <h2><?php esc_html_e( 'Analyzing conversation...', 'converge-applybyvoice' ); ?></h2>
                                <p><?php esc_html_e( 'AI insights are being generated. This page will update automatically.', 'converge-applybyvoice' ); ?></p>
                            </div>
                        </div>
                    <?php elseif ( $conv['insightsStatus'] === 'failed' ) : ?>
                        <div class="converge-card">
                            <p><?php esc_html_e( 'Insight generation failed for this conversation.', 'converge-applybyvoice' ); ?></p>
                        </div>
                    <?php elseif ( $conv['insightsStatus'] === 'skipped' ) : ?>
                        <div class="converge-card">
                            <p><?php esc_html_e( 'This conversation was too short for insights to be generated.', 'converge-applybyvoice' ); ?></p>
                        </div>
                    <?php endif; ?>

                    <!-- Screening Data -->
                    <?php if ( $screening ) :
                        $screening_items = array();

                        if ( ! empty( $screening['availability'] ) ) {
                            $screening_items[] = array(
                                'label' => __( 'Availability', 'converge-applybyvoice' ),
                                'value' => $screening['availability'],
                            );
                        }
                        if ( ! empty( $screening['salaryRaw'] ) ) {
                            $screening_items[] = array(
                                'label' => __( 'Salary Expectation', 'converge-applybyvoice' ),
                                'value' => $screening['salaryRaw'],
                            );
                        } elseif ( ! empty( $screening['salaryMin'] ) ) {
                            $cur = $screening['salaryCurrency'] ?? 'USD';
                            $val = number_format( $screening['salaryMin'] );
                            if ( ! empty( $screening['salaryMax'] ) ) {
                                $val .= ' - ' . number_format( $screening['salaryMax'] );
                            }
                            $screening_items[] = array(
                                'label' => __( 'Salary Expectation', 'converge-applybyvoice' ),
                                'value' => $cur . ' ' . $val,
                            );
                        }
                        if ( ! empty( $screening['workAuthStatus'] ) ) {
                            $auth_label = ucfirst( str_replace( '_', ' ', $screening['workAuthStatus'] ) );
                            if ( ! empty( $screening['requiresSponsorship'] ) ) {
                                $auth_label .= ' (' . __( 'requires sponsorship', 'converge-applybyvoice' ) . ')';
                            }
                            $screening_items[] = array(
                                'label' => __( 'Work Authorization', 'converge-applybyvoice' ),
                                'value' => $auth_label,
                            );
                        }
                        if ( ! empty( $screening['noticePeriodRaw'] ) ) {
                            $screening_items[] = array(
                                'label' => __( 'Notice Period', 'converge-applybyvoice' ),
                                'value' => $screening['noticePeriodRaw'],
                            );
                        } elseif ( ! empty( $screening['noticePeriodDays'] ) ) {
                            $screening_items[] = array(
                                'label' => __( 'Notice Period', 'converge-applybyvoice' ),
                                'value' => $screening['noticePeriodDays'] . ' ' . __( 'days', 'converge-applybyvoice' ),
                            );
                        }
                        if ( ! empty( $screening['currentLocation'] ) ) {
                            $loc = $screening['currentLocation'];
                            if ( isset( $screening['relocationWilling'] ) ) {
                                $loc .= $screening['relocationWilling']
                                    ? ' (' . __( 'open to relocation', 'converge-applybyvoice' ) . ')'
                                    : ' (' . __( 'not open to relocation', 'converge-applybyvoice' ) . ')';
                            }
                            $screening_items[] = array(
                                'label' => __( 'Location', 'converge-applybyvoice' ),
                                'value' => $loc,
                            );
                        }
                        if ( ! empty( $screening['currentEmploymentStatus'] ) ) {
                            $emp = ucfirst( $screening['currentEmploymentStatus'] );
                            if ( ! empty( $screening['currentEmployer'] ) ) {
                                $emp .= ' at ' . $screening['currentEmployer'];
                            }
                            if ( ! empty( $screening['currentJobTitle'] ) ) {
                                $emp .= ' (' . $screening['currentJobTitle'] . ')';
                            }
                            $screening_items[] = array(
                                'label' => __( 'Employment Status', 'converge-applybyvoice' ),
                                'value' => $emp,
                            );
                        }
                        if ( ! empty( $screening['reasonForLeaving'] ) ) {
                            $screening_items[] = array(
                                'label' => __( 'Reason for Leaving', 'converge-applybyvoice' ),
                                'value' => $screening['reasonForLeaving'],
                            );
                        }
                    ?>
                        <?php if ( ! empty( $screening_items ) ) : ?>
                            <div class="converge-card">
                                <h2><?php esc_html_e( 'Screening Data', 'converge-applybyvoice' ); ?></h2>
                                <table class="widefat striped">
                                    <tbody>
                                        <?php foreach ( $screening_items as $item ) : ?>
                                            <tr>
                                                <th style="width: 180px;"><?php echo esc_html( $item['label'] ); ?></th>
                                                <td><?php echo esc_html( $item['value'] ); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    <?php endif; ?>

                    <!-- Transcript -->
                    <?php if ( ! empty( $transcript ) ) : ?>
                        <div class="converge-card">
                            <h2><?php esc_html_e( 'Conversation Transcript', 'converge-applybyvoice' ); ?></h2>
                            <div class="converge-transcript">
                                <?php foreach ( $transcript as $msg ) :
                                    $role_class = $msg['role'] === 'ai' ? 'converge-msg-ai' : 'converge-msg-candidate';
                                    $role_label = $msg['role'] === 'ai'
                                        ? __( 'AI', 'converge-applybyvoice' )
                                        : ( $candidate['name'] ?? __( 'Candidate', 'converge-applybyvoice' ) );
                                    $time_str = '';
                                    if ( ! empty( $msg['timestampMs'] ) ) {
                                        $secs = intval( $msg['timestampMs'] / 1000 );
                                        $time_str = sprintf( '%d:%02d', floor( $secs / 60 ), $secs % 60 );
                                    }
                                ?>
                                    <div class="converge-msg <?php echo esc_attr( $role_class ); ?>">
                                        <div class="converge-msg-header">
                                            <span class="converge-msg-role"><?php echo esc_html( $role_label ); ?></span>
                                            <?php if ( $time_str ) : ?>
                                                <span class="converge-msg-time"><?php echo esc_html( $time_str ); ?></span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="converge-msg-content"><?php echo esc_html( $msg['content'] ); ?></div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endif; ?>

                </div>

                <!-- Right Column: Sidebar -->
                <div class="converge-review-sidebar">

                    <!-- Candidate Info -->
                    <?php if ( $candidate ) : ?>
                        <div class="converge-card">
                            <h2><?php esc_html_e( 'Candidate', 'converge-applybyvoice' ); ?></h2>
                            <p><strong><?php echo esc_html( $candidate['name'] ); ?></strong></p>
                            <?php if ( ! empty( $candidate['email'] ) ) : ?>
                                <p><?php echo esc_html( $candidate['email'] ); ?></p>
                            <?php endif; ?>
                            <?php if ( ! empty( $candidate['phone'] ) ) : ?>
                                <p><?php echo esc_html( $candidate['phone'] ); ?></p>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                    <!-- Conversation Details -->
                    <div class="converge-card">
                        <h2><?php esc_html_e( 'Details', 'converge-applybyvoice' ); ?></h2>
                        <table class="converge-detail-table">
                            <tr>
                                <th><?php esc_html_e( 'Job', 'converge-applybyvoice' ); ?></th>
                                <td><?php echo esc_html( $job['title'] ); ?></td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Status', 'converge-applybyvoice' ); ?></th>
                                <td>
                                    <span class="converge-status-badge converge-status-<?php echo esc_attr( $conv['status'] ); ?>">
                                        <?php echo esc_html( ucfirst( str_replace( '_', ' ', $conv['status'] ) ) ); ?>
                                    </span>
                                </td>
                            </tr>
                            <?php if ( $duration_str ) : ?>
                                <tr>
                                    <th><?php esc_html_e( 'Duration', 'converge-applybyvoice' ); ?></th>
                                    <td><?php echo esc_html( $duration_str ); ?></td>
                                </tr>
                            <?php endif; ?>
                            <tr>
                                <th><?php esc_html_e( 'Date', 'converge-applybyvoice' ); ?></th>
                                <td><?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $conv['createdAt'] ) ) ); ?></td>
                            </tr>
                            <?php if ( ! empty( $conv['channel'] ) ) : ?>
                                <tr>
                                    <th><?php esc_html_e( 'Channel', 'converge-applybyvoice' ); ?></th>
                                    <td><?php echo esc_html( ucfirst( $conv['channel'] ) ); ?></td>
                                </tr>
                            <?php endif; ?>
                            <?php if ( ! empty( $conv['language'] ) && $conv['language'] !== 'en' ) : ?>
                                <tr>
                                    <th><?php esc_html_e( 'Language', 'converge-applybyvoice' ); ?></th>
                                    <td><?php echo esc_html( strtoupper( $conv['language'] ) ); ?></td>
                                </tr>
                            <?php endif; ?>
                            <?php if ( ! empty( $conv['connectionQuality'] ) ) : ?>
                                <tr>
                                    <th><?php esc_html_e( 'Connection', 'converge-applybyvoice' ); ?></th>
                                    <td>
                                        <?php
                                        $quality = $conv['connectionQuality'];
                                        $q_class = $quality === 'good' ? 'converge-quality-good' : ( $quality === 'fair' ? 'converge-quality-fair' : 'converge-quality-poor' );
                                        ?>
                                        <span class="converge-connection-badge <?php echo esc_attr( $q_class ); ?>">
                                            <?php echo esc_html( ucfirst( $quality ) ); ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </table>
                    </div>

                    <!-- Consent & Accommodations -->
                    <?php if ( ! empty( $conv['consentRecording'] ) || ! empty( $conv['accommodationRequested'] ) ) : ?>
                        <div class="converge-card">
                            <h2><?php esc_html_e( 'Consent & Accommodations', 'converge-applybyvoice' ); ?></h2>
                            <?php if ( ! empty( $conv['consentRecording'] ) ) : ?>
                                <p>
                                    <span class="dashicons dashicons-yes-alt" style="color:#46b450;"></span>
                                    <?php esc_html_e( 'Recording consent given', 'converge-applybyvoice' ); ?>
                                </p>
                            <?php endif; ?>
                            <?php if ( ! empty( $conv['consentDataProcessing'] ) ) : ?>
                                <p>
                                    <span class="dashicons dashicons-yes-alt" style="color:#46b450;"></span>
                                    <?php esc_html_e( 'Data processing consent given', 'converge-applybyvoice' ); ?>
                                </p>
                            <?php endif; ?>
                            <?php if ( ! empty( $conv['accommodationRequested'] ) ) : ?>
                                <div style="margin-top: 8px; padding-top: 8px; border-top: 1px solid #eee;">
                                    <p><strong><?php esc_html_e( 'Accommodations Requested:', 'converge-applybyvoice' ); ?></strong></p>
                                    <?php if ( ! empty( $conv['accommodationTypes'] ) && is_array( $conv['accommodationTypes'] ) ) : ?>
                                        <ul class="converge-insight-list">
                                            <?php foreach ( $conv['accommodationTypes'] as $acc_type ) : ?>
                                                <li><?php echo esc_html( ucfirst( str_replace( '_', ' ', $acc_type ) ) ); ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    <?php endif; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                    <!-- Questions Candidate Asked -->
                    <?php if ( $insights && ! empty( $insights['candidateQuestions'] ) ) : ?>
                        <div class="converge-card">
                            <h2><?php esc_html_e( 'Candidate Questions', 'converge-applybyvoice' ); ?></h2>
                            <ul class="converge-insight-list">
                                <?php foreach ( $insights['candidateQuestions'] as $q ) :
                                    $q_text = is_array( $q ) ? ( $q['question'] ?? $q['text'] ?? '' ) : $q;
                                ?>
                                    <li><?php echo esc_html( $q_text ); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <!-- Intent Signals -->
                    <?php if ( $insights && ! empty( $insights['intentSignals'] ) ) :
                        $intent = $insights['intentSignals'];
                    ?>
                        <div class="converge-card">
                            <h2><?php esc_html_e( 'Intent Signals', 'converge-applybyvoice' ); ?></h2>
                            <?php if ( is_array( $intent ) ) : ?>
                                <?php if ( ! empty( $intent['interest_level'] ) ) : ?>
                                    <p><strong><?php esc_html_e( 'Interest:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( ucfirst( $intent['interest_level'] ) ); ?></p>
                                <?php endif; ?>
                                <?php if ( ! empty( $intent['timeline'] ) ) : ?>
                                    <p><strong><?php esc_html_e( 'Timeline:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $intent['timeline'] ); ?></p>
                                <?php endif; ?>
                                <?php if ( ! empty( $intent['competing_offers'] ) ) : ?>
                                    <p><strong><?php esc_html_e( 'Competing offers:', 'converge-applybyvoice' ); ?></strong> <?php echo esc_html( $intent['competing_offers'] ); ?></p>
                                <?php endif; ?>
                            <?php else : ?>
                                <p><?php echo esc_html( $intent ); ?></p>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                </div>
            </div>

        <?php endif; ?>
    <?php endif; ?>
</div>
