<?php
/**
 * Template: Native Job Detail page for Converge WordPress plugin.
 * Shows candidates/conversations for a single job, with 6 config tabs matching web app.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

$auth = new Converge_Auth();
$connected = $auth->is_connected();
$job_id = isset( $_GET['job_id'] ) ? sanitize_text_field( wp_unslash( $_GET['job_id'] ) ) : '';
$active_tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'candidates';
$valid_tabs = array( 'candidates', 'details', 'conversation', 'screening', 'knowledge', 'ai' );
if ( ! in_array( $active_tab, $valid_tabs, true ) ) {
    $active_tab = 'candidates';
}
?>

<div class="wrap">
    <?php if ( ! $connected ) : ?>
        <h1><?php esc_html_e( 'Job Detail', 'converge-applybyvoice' ); ?></h1>
        <div class="converge-card converge-connect-cta">
            <h2><?php esc_html_e( 'Connect Your Account', 'converge-applybyvoice' ); ?></h2>
            <p><?php esc_html_e( 'Connect your Converge account to view job details.', 'converge-applybyvoice' ); ?></p>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=converge-settings' ) ); ?>" class="button button-primary">
                <?php esc_html_e( 'Connect Account', 'converge-applybyvoice' ); ?>
            </a>
        </div>
    <?php elseif ( empty( $job_id ) ) : ?>
        <h1><?php esc_html_e( 'Job Detail', 'converge-applybyvoice' ); ?></h1>
        <div class="notice notice-error"><p><?php esc_html_e( 'No job specified.', 'converge-applybyvoice' ); ?></p></div>
    <?php else :
        $api = new Converge_API( $auth );
        $result = $api->get_job_detail( $job_id );

        if ( is_wp_error( $result ) ) : ?>
            <h1><?php esc_html_e( 'Job Detail', 'converge-applybyvoice' ); ?></h1>
            <div class="notice notice-error"><p><?php echo esc_html( $result->get_error_message() ); ?></p></div>
        <?php else :
            $job = $result['job'];
            $candidates = $result['candidates'] ?? array();
            $entry_methods = $result['entryMethods'] ?? array();
            $completeness = $job['completeness'] ?? array();
            $kb = $job['jobKnowledgeBase'] ?? array();
            $sr = $job['screeningRequirements'] ?? array();
            $competencies = $job['parsedCompetencies'] ?? array();
            $conv_config = $job['conversationConfig'] ?? array();

            // Find an active entry method for the apply link
            $apply_url = '';
            $active_entry = null;
            foreach ( $entry_methods as $em ) {
                if ( $em['isActive'] ) {
                    $apply_url = CONVERGE_CANDIDATE_BASE . '/c/' . $em['publicId'];
                    $active_entry = $em;
                    break;
                }
            }

            $base_url = admin_url( 'admin.php?page=converge-job-detail&job_id=' . urlencode( $job['id'] ) );

            // Status colors
            $status_colors = array(
                'active'    => '#0073aa',
                'draft'     => '#646970',
                'paused'    => '#e69500',
                'closed'    => '#dc3545',
            );
            $status_color = $status_colors[ $job['status'] ] ?? '#646970';
        ?>

            <!-- Job Header -->
            <div class="converge-job-header">
                <div class="converge-job-header-left">
                    <h1 class="wp-heading-inline"><?php echo esc_html( $job['title'] ); ?></h1>
                    <span class="converge-status-pill" style="background: <?php echo esc_attr( $status_color ); ?>;">
                        <?php echo esc_html( ucfirst( $job['status'] ) ); ?>
                    </span>
                </div>
                <div class="converge-job-header-actions">
                    <?php if ( $job['status'] === 'draft' ) : ?>
                        <button type="button" class="button button-primary converge-job-status-btn" data-job-id="<?php echo esc_attr( $job['id'] ); ?>" data-status="active">
                            <?php esc_html_e( 'Publish', 'converge-applybyvoice' ); ?>
                        </button>
                    <?php elseif ( $job['status'] === 'active' ) : ?>
                        <button type="button" class="button converge-job-status-btn" data-job-id="<?php echo esc_attr( $job['id'] ); ?>" data-status="paused">
                            <?php esc_html_e( 'Pause', 'converge-applybyvoice' ); ?>
                        </button>
                        <button type="button" class="button converge-job-status-btn" data-job-id="<?php echo esc_attr( $job['id'] ); ?>" data-status="closed">
                            <?php esc_html_e( 'Close', 'converge-applybyvoice' ); ?>
                        </button>
                    <?php elseif ( $job['status'] === 'paused' ) : ?>
                        <button type="button" class="button button-primary converge-job-status-btn" data-job-id="<?php echo esc_attr( $job['id'] ); ?>" data-status="active">
                            <?php esc_html_e( 'Reactivate', 'converge-applybyvoice' ); ?>
                        </button>
                        <button type="button" class="button converge-job-status-btn" data-job-id="<?php echo esc_attr( $job['id'] ); ?>" data-status="closed">
                            <?php esc_html_e( 'Close', 'converge-applybyvoice' ); ?>
                        </button>
                    <?php elseif ( $job['status'] === 'closed' ) : ?>
                        <button type="button" class="button converge-job-status-btn" data-job-id="<?php echo esc_attr( $job['id'] ); ?>" data-status="draft">
                            <?php esc_html_e( 'Reopen as Draft', 'converge-applybyvoice' ); ?>
                        </button>
                    <?php endif; ?>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=converge-jobs' ) ); ?>" class="button">
                        &larr; <?php esc_html_e( 'All Jobs', 'converge-applybyvoice' ); ?>
                    </a>
                </div>
            </div>
            <hr class="wp-header-end">

            <!-- 6-Tab Navigation -->
            <h2 class="nav-tab-wrapper converge-job-tabs">
                <a href="<?php echo esc_url( $base_url . '&tab=candidates' ); ?>"
                   class="nav-tab <?php echo $active_tab === 'candidates' ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Candidates', 'converge-applybyvoice' ); ?>
                    <?php if ( count( $candidates ) > 0 ) : ?>
                        <span class="converge-tab-count"><?php echo esc_html( count( $candidates ) ); ?></span>
                    <?php endif; ?>
                </a>
                <a href="<?php echo esc_url( $base_url . '&tab=details' ); ?>"
                   class="nav-tab <?php echo $active_tab === 'details' ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Details', 'converge-applybyvoice' ); ?>
                </a>
                <a href="<?php echo esc_url( $base_url . '&tab=conversation' ); ?>"
                   class="nav-tab <?php echo $active_tab === 'conversation' ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Conversation', 'converge-applybyvoice' ); ?>
                </a>
                <a href="<?php echo esc_url( $base_url . '&tab=screening' ); ?>"
                   class="nav-tab <?php echo $active_tab === 'screening' ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Screening', 'converge-applybyvoice' ); ?>
                </a>
                <a href="<?php echo esc_url( $base_url . '&tab=knowledge' ); ?>"
                   class="nav-tab <?php echo $active_tab === 'knowledge' ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Knowledge', 'converge-applybyvoice' ); ?>
                </a>
                <a href="<?php echo esc_url( $base_url . '&tab=ai' ); ?>"
                   class="nav-tab <?php echo $active_tab === 'ai' ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'AI', 'converge-applybyvoice' ); ?>
                </a>
            </h2>

            <!-- Tab: Candidates -->
            <?php if ( $active_tab === 'candidates' ) : ?>

                <div class="converge-job-layout">
                    <div class="converge-job-main">

                        <!-- Status filter tabs -->
                        <div class="converge-filter-bar">
                            <div class="converge-status-filters" id="converge-candidate-filters" data-job-id="<?php echo esc_attr( $job['id'] ); ?>">
                                <?php
                                $status_counts = array( 'all' => count( $candidates ), 'new' => 0, 'viewed' => 0, 'shortlisted' => 0, 'archived' => 0 );
                                foreach ( $candidates as $c ) {
                                    $rs = $c['reviewStatus'] ?? 'new';
                                    if ( isset( $status_counts[ $rs ] ) ) {
                                        $status_counts[ $rs ]++;
                                    }
                                }
                                $filters = array(
                                    'all'         => __( 'All', 'converge-applybyvoice' ),
                                    'new'         => __( 'New', 'converge-applybyvoice' ),
                                    'viewed'      => __( 'Viewed', 'converge-applybyvoice' ),
                                    'shortlisted' => __( 'Shortlisted', 'converge-applybyvoice' ),
                                    'archived'    => __( 'Archived', 'converge-applybyvoice' ),
                                );
                                foreach ( $filters as $fkey => $flabel ) : ?>
                                    <button type="button" class="converge-filter-pill <?php echo $fkey === 'all' ? 'is-active' : ''; ?>" data-filter="<?php echo esc_attr( $fkey ); ?>">
                                        <?php echo esc_html( $flabel ); ?>
                                        <span class="converge-filter-count"><?php echo esc_html( $status_counts[ $fkey ] ); ?></span>
                                    </button>
                                <?php endforeach; ?>
                            </div>
                            <div class="converge-sort-wrap">
                                <select id="converge-candidate-sort">
                                    <option value="newest"><?php esc_html_e( 'Newest', 'converge-applybyvoice' ); ?></option>
                                    <option value="oldest"><?php esc_html_e( 'Oldest', 'converge-applybyvoice' ); ?></option>
                                    <option value="name"><?php esc_html_e( 'By Name', 'converge-applybyvoice' ); ?></option>
                                </select>
                            </div>
                        </div>

                        <!-- Candidates List -->
                        <div id="converge-candidates-list">
                            <?php if ( empty( $candidates ) ) : ?>
                                <div class="converge-card">
                                    <p><?php esc_html_e( 'No candidates yet. Share the apply link to start receiving voice applications.', 'converge-applybyvoice' ); ?></p>
                                </div>
                            <?php else : ?>
                                <?php foreach ( $candidates as $c ) :
                                    $duration_str = '';
                                    if ( ! empty( $c['duration'] ) ) {
                                        $duration_str = sprintf( '%d:%02d', floor( $c['duration'] / 60 ), $c['duration'] % 60 );
                                    }
                                    $review_status = $c['reviewStatus'] ?? 'new';
                                    $is_new = ( $review_status === 'new' );
                                    $is_shortlisted = ( $review_status === 'shortlisted' );
                                    $is_archived = ( $review_status === 'archived' );
                                    $total_competencies = count( $competencies );
                                    $ci_count = $c['competencyCoverage'] ?? 0;
                                    $summary = $c['summary'] ?? '';
                                    $strengths = $c['strengths'] ?? array();
                                    $areas_to_explore = $c['areasToExplore'] ?? array();
                                    $card_classes = 'converge-candidate-card';
                                    if ( $is_archived ) $card_classes .= ' converge-candidate-archived';
                                    if ( $is_shortlisted ) $card_classes .= ' converge-candidate-shortlisted';
                                    $detail_url = admin_url( 'admin.php?page=converge-conversation&conversation_id=' . urlencode( $c['conversationId'] ) . '&job_id=' . urlencode( $job['id'] ) );
                                ?>
                                    <div class="<?php echo esc_attr( $card_classes ); ?>"
                                         data-review-status="<?php echo esc_attr( $review_status ); ?>"
                                         data-conversation-id="<?php echo esc_attr( $c['conversationId'] ); ?>"
                                         data-name="<?php echo esc_attr( strtolower( $c['name'] ) ); ?>"
                                         data-date="<?php echo esc_attr( $c['createdAt'] ); ?>">

                                        <!-- Status indicator -->
                                        <div class="converge-candidate-indicator">
                                            <?php if ( $is_new ) : ?>
                                                <span class="converge-new-dot" title="<?php esc_attr_e( 'New', 'converge-applybyvoice' ); ?>"></span>
                                            <?php elseif ( $is_shortlisted ) : ?>
                                                <span class="dashicons dashicons-star-filled converge-indicator-shortlisted" title="<?php esc_attr_e( 'Shortlisted', 'converge-applybyvoice' ); ?>"></span>
                                            <?php elseif ( $is_archived ) : ?>
                                                <span class="dashicons dashicons-archive converge-indicator-archived" title="<?php esc_attr_e( 'Archived', 'converge-applybyvoice' ); ?>"></span>
                                            <?php else : ?>
                                                <span class="dashicons dashicons-visibility converge-indicator-viewed" title="<?php esc_attr_e( 'Viewed', 'converge-applybyvoice' ); ?>"></span>
                                            <?php endif; ?>
                                        </div>

                                        <!-- Main content -->
                                        <div class="converge-candidate-body">
                                            <!-- Header: name, email, time -->
                                            <div class="converge-candidate-header">
                                                <div class="converge-candidate-identity">
                                                    <a href="<?php echo esc_url( $detail_url ); ?>" class="converge-candidate-name">
                                                        <?php echo esc_html( $c['name'] ); ?>
                                                    </a>
                                                    <?php if ( ! empty( $c['email'] ) ) : ?>
                                                        <span class="converge-candidate-email"><?php echo esc_html( $c['email'] ); ?></span>
                                                    <?php endif; ?>
                                                </div>
                                                <span class="converge-candidate-time" data-timestamp="<?php echo esc_attr( $c['createdAt'] ); ?>">
                                                    <?php echo esc_html( human_time_diff( strtotime( $c['createdAt'] ), current_time( 'timestamp' ) ) . ' ' . __( 'ago', 'converge-applybyvoice' ) ); ?>
                                                </span>
                                            </div>

                                            <!-- Meta row: duration + competency badge -->
                                            <div class="converge-candidate-meta">
                                                <?php if ( $duration_str ) : ?>
                                                    <span class="converge-candidate-duration"><?php echo esc_html( $duration_str ); ?> <?php esc_html_e( 'conversation', 'converge-applybyvoice' ); ?></span>
                                                <?php elseif ( $c['status'] === 'in_progress' ) : ?>
                                                    <span class="converge-candidate-duration"><?php esc_html_e( 'In progress', 'converge-applybyvoice' ); ?></span>
                                                <?php else : ?>
                                                    <span class="converge-candidate-duration"><?php esc_html_e( 'N/A conversation', 'converge-applybyvoice' ); ?></span>
                                                <?php endif; ?>

                                                <?php if ( $ci_count > 0 && $total_competencies > 0 ) : ?>
                                                    <span class="converge-competency-badge">
                                                        <span class="dashicons dashicons-welcome-learn-more"></span>
                                                        <?php printf(
                                                            /* translators: 1: discussed count, 2: total count */
                                                            esc_html__( '%1$d of %2$d competencies', 'converge-applybyvoice' ),
                                                            intval( $ci_count ),
                                                            intval( $total_competencies )
                                                        ); ?>
                                                    </span>
                                                <?php endif; ?>
                                            </div>

                                            <!-- Summary (2-line clamp) -->
                                            <?php if ( ! empty( $summary ) ) : ?>
                                                <p class="converge-candidate-summary"><?php echo esc_html( $summary ); ?></p>
                                            <?php endif; ?>

                                            <!-- Key Strengths -->
                                            <?php if ( ! empty( $strengths ) ) : ?>
                                                <div class="converge-candidate-insights-section">
                                                    <span class="converge-insights-label converge-strengths-label">
                                                        <span class="dashicons dashicons-yes-alt"></span>
                                                        <?php esc_html_e( 'Key strengths', 'converge-applybyvoice' ); ?>
                                                    </span>
                                                    <ul class="converge-candidate-insights-list">
                                                        <?php foreach ( array_slice( $strengths, 0, 2 ) as $s ) : ?>
                                                            <li><?php echo esc_html( $s ); ?></li>
                                                        <?php endforeach; ?>
                                                    </ul>
                                                </div>
                                            <?php endif; ?>

                                            <!-- Areas to Explore -->
                                            <?php if ( ! empty( $areas_to_explore ) ) : ?>
                                                <div class="converge-candidate-insights-section">
                                                    <span class="converge-insights-label converge-explore-label">
                                                        <span class="dashicons dashicons-search"></span>
                                                        <?php esc_html_e( 'Areas to explore', 'converge-applybyvoice' ); ?>
                                                    </span>
                                                    <ul class="converge-candidate-insights-list">
                                                        <?php foreach ( array_slice( $areas_to_explore, 0, 2 ) as $a ) : ?>
                                                            <li><?php echo esc_html( $a ); ?></li>
                                                        <?php endforeach; ?>
                                                    </ul>
                                                </div>
                                            <?php endif; ?>

                                            <!-- Action buttons -->
                                            <div class="converge-candidate-actions">
                                                <?php if ( ! $is_shortlisted ) : ?>
                                                    <button type="button" class="button button-small converge-review-action" data-conversation-id="<?php echo esc_attr( $c['conversationId'] ); ?>" data-action="shortlisted">
                                                        <span class="dashicons dashicons-star-empty"></span>
                                                        <?php esc_html_e( 'Shortlist', 'converge-applybyvoice' ); ?>
                                                    </button>
                                                <?php else : ?>
                                                    <button type="button" class="button button-small converge-review-action is-active-shortlist" data-conversation-id="<?php echo esc_attr( $c['conversationId'] ); ?>" data-action="viewed">
                                                        <span class="dashicons dashicons-star-filled"></span>
                                                        <?php esc_html_e( 'Shortlisted', 'converge-applybyvoice' ); ?>
                                                    </button>
                                                <?php endif; ?>

                                                <?php if ( ! $is_archived ) : ?>
                                                    <button type="button" class="button button-small converge-review-action" data-conversation-id="<?php echo esc_attr( $c['conversationId'] ); ?>" data-action="archived">
                                                        <span class="dashicons dashicons-archive"></span>
                                                        <?php esc_html_e( 'Archive', 'converge-applybyvoice' ); ?>
                                                    </button>
                                                <?php else : ?>
                                                    <button type="button" class="button button-small converge-review-action" data-conversation-id="<?php echo esc_attr( $c['conversationId'] ); ?>" data-action="viewed">
                                                        <span class="dashicons dashicons-undo"></span>
                                                        <?php esc_html_e( 'Restore', 'converge-applybyvoice' ); ?>
                                                    </button>
                                                <?php endif; ?>

                                                <a href="<?php echo esc_url( $detail_url ); ?>" class="button button-primary button-small">
                                                    <?php esc_html_e( 'View Details', 'converge-applybyvoice' ); ?>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Sidebar -->
                    <div class="converge-job-sidebar">
                        <!-- Stats Card -->
                        <div class="converge-card">
                            <h2><?php esc_html_e( 'Stats', 'converge-applybyvoice' ); ?></h2>
                            <table class="converge-detail-table">
                                <tr>
                                    <th><?php esc_html_e( 'Total', 'converge-applybyvoice' ); ?></th>
                                    <td><?php echo esc_html( count( $candidates ) ); ?></td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e( 'Completed', 'converge-applybyvoice' ); ?></th>
                                    <td><?php echo esc_html( count( array_filter( $candidates, function( $c ) { return $c['status'] === 'completed'; } ) ) ); ?></td>
                                </tr>
                                <?php
                                $total_c = count( $candidates );
                                $completed_c = count( array_filter( $candidates, function( $c ) { return $c['status'] === 'completed'; } ) );
                                $rate = $total_c > 0 ? round( ( $completed_c / $total_c ) * 100 ) : 0;
                                ?>
                                <tr>
                                    <th><?php esc_html_e( 'Rate', 'converge-applybyvoice' ); ?></th>
                                    <td><?php echo esc_html( $rate . '%' ); ?></td>
                                </tr>
                            </table>
                        </div>

                        <!-- Share This Job Card -->
                        <?php if ( ! empty( $entry_methods ) ) : ?>
                            <div class="converge-card">
                                <h2><?php esc_html_e( 'Share This Job', 'converge-applybyvoice' ); ?></h2>
                                <div class="converge-share-tabs">
                                    <button type="button" class="converge-share-tab is-active" data-share="link"><?php esc_html_e( 'Link', 'converge-applybyvoice' ); ?></button>
                                    <button type="button" class="converge-share-tab" data-share="qr"><?php esc_html_e( 'QR', 'converge-applybyvoice' ); ?></button>
                                    <button type="button" class="converge-share-tab" data-share="embed"><?php esc_html_e( 'Embed', 'converge-applybyvoice' ); ?></button>
                                    <button type="button" class="converge-share-tab" data-share="widget"><?php esc_html_e( 'Widget', 'converge-applybyvoice' ); ?></button>
                                </div>
                                <?php
                                $org_slug = get_option( 'converge_org_slug', '' );
                                ?>
                                <!-- Link panel -->
                                <div class="converge-share-panel is-active" data-share-panel="link">
                                    <?php if ( $apply_url ) : ?>
                                        <div class="converge-copy-field">
                                            <input type="text" value="<?php echo esc_attr( $apply_url ); ?>" readonly>
                                            <button type="button" class="button button-small converge-copy-btn" data-copy="<?php echo esc_attr( $apply_url ); ?>">
                                                <?php esc_html_e( 'Copy', 'converge-applybyvoice' ); ?>
                                            </button>
                                        </div>
                                        <a href="<?php echo esc_url( $apply_url ); ?>" target="_blank" class="converge-preview-link">
                                            <?php esc_html_e( 'Open Preview', 'converge-applybyvoice' ); ?> <span class="dashicons dashicons-external" style="font-size: 14px; width: 14px; height: 14px;"></span>
                                        </a>
                                    <?php else : ?>
                                        <p class="description"><?php esc_html_e( 'No active entry method.', 'converge-applybyvoice' ); ?></p>
                                    <?php endif; ?>
                                </div>
                                <!-- QR panel -->
                                <div class="converge-share-panel" data-share-panel="qr">
                                    <?php if ( $apply_url ) : ?>
                                        <div id="converge-qr-code" class="converge-qr-container" data-url="<?php echo esc_attr( $apply_url ); ?>"></div>
                                        <button type="button" class="button button-small" id="converge-download-qr" style="margin-top: 8px;">
                                            <?php esc_html_e( 'Download PNG', 'converge-applybyvoice' ); ?>
                                        </button>
                                    <?php else : ?>
                                        <p class="description"><?php esc_html_e( 'No active entry method.', 'converge-applybyvoice' ); ?></p>
                                    <?php endif; ?>
                                </div>
                                <!-- Embed panel -->
                                <div class="converge-share-panel" data-share-panel="embed">
                                    <?php if ( $active_entry ) :
                                        $embed_code = '<script src="' . CONVERGE_API_BASE . '/apply.js" data-converge-id="' . $active_entry['publicId'] . '" data-mode="modal" data-domain="' . CONVERGE_CANDIDATE_BASE . '"></script>';
                                    ?>
                                        <div class="converge-code-snippet">
                                            <code><?php echo esc_html( $embed_code ); ?></code>
                                        </div>
                                        <button type="button" class="button button-small converge-copy-btn" data-copy="<?php echo esc_attr( $embed_code ); ?>" style="margin-top: 6px;">
                                            <?php esc_html_e( 'Copy Code', 'converge-applybyvoice' ); ?>
                                        </button>
                                    <?php else : ?>
                                        <p class="description"><?php esc_html_e( 'No active entry method.', 'converge-applybyvoice' ); ?></p>
                                    <?php endif; ?>
                                </div>
                                <!-- Widget panel -->
                                <div class="converge-share-panel" data-share-panel="widget">
                                    <?php if ( $org_slug ) :
                                        $widget_code = '<script src="' . CONVERGE_API_BASE . '/widget.js" data-org="' . $org_slug . '" data-mode="float" data-domain="' . CONVERGE_CANDIDATE_BASE . '"></script>';
                                    ?>
                                        <div class="converge-code-snippet">
                                            <code><?php echo esc_html( $widget_code ); ?></code>
                                        </div>
                                        <button type="button" class="button button-small converge-copy-btn" data-copy="<?php echo esc_attr( $widget_code ); ?>" style="margin-top: 6px;">
                                            <?php esc_html_e( 'Copy Code', 'converge-applybyvoice' ); ?>
                                        </button>
                                    <?php else : ?>
                                        <p class="description"><?php esc_html_e( 'Organization slug not available.', 'converge-applybyvoice' ); ?></p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- Setup Checklist (compact) -->
                        <div class="converge-card">
                            <h2><?php esc_html_e( 'Setup', 'converge-applybyvoice' ); ?>
                                <?php
                                $c_count = $completeness['configuredCount'] ?? 0;
                                $c_total = $completeness['totalChecks'] ?? 7;
                                $c_tier  = $completeness['tier'] ?? 'incomplete';
                                ?>
                                <span class="converge-setup-indicator converge-setup-<?php echo esc_attr( $c_tier ); ?>" style="font-size: 12px; margin-left: 8px;">
                                    <?php echo esc_html( $c_count . '/' . $c_total ); ?>
                                </span>
                            </h2>
                            <?php
                            $checklist = array(
                                array( 'key' => 'hasCompetencies', 'label' => __( 'Competencies', 'converge-applybyvoice' ), 'tab' => 'details' ),
                                array( 'key' => 'hasLocation',     'label' => __( 'Location', 'converge-applybyvoice' ),     'tab' => 'details' ),
                                array( 'key' => 'hasDescription',  'label' => __( 'Description', 'converge-applybyvoice' ),  'tab' => 'details' ),
                                array( 'key' => 'hasSalaryRange',  'label' => __( 'Salary', 'converge-applybyvoice' ),       'tab' => 'knowledge' ),
                                array( 'key' => 'hasScreening',    'label' => __( 'Screening', 'converge-applybyvoice' ),    'tab' => 'screening' ),
                                array( 'key' => 'hasKnowledgeBase','label' => __( 'Knowledge', 'converge-applybyvoice' ),    'tab' => 'knowledge' ),
                                array( 'key' => 'hasAIConfig',     'label' => __( 'AI Config', 'converge-applybyvoice' ),    'tab' => 'ai' ),
                            );
                            foreach ( $checklist as $item ) :
                                $done = ! empty( $completeness[ $item['key'] ] );
                            ?>
                                <div class="converge-checklist-row">
                                    <span class="dashicons <?php echo $done ? 'dashicons-yes-alt converge-check-done' : 'dashicons-minus converge-check-missing'; ?>"></span>
                                    <a href="<?php echo esc_url( $base_url . '&tab=' . $item['tab'] ); ?>"><?php echo esc_html( $item['label'] ); ?></a>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>

            <!-- Tab: Details -->
            <?php elseif ( $active_tab === 'details' ) : ?>

                <!-- Basic Information Form -->
                <div class="converge-card" id="section-basic">
                    <h2><?php esc_html_e( 'Basic Information', 'converge-applybyvoice' ); ?></h2>
                    <form class="converge-job-form" data-section="basic" data-job-id="<?php echo esc_attr( $job['id'] ); ?>">
                        <table class="form-table">
                            <tr>
                                <th scope="row"><label for="job-title"><?php esc_html_e( 'Title', 'converge-applybyvoice' ); ?></label></th>
                                <td><input type="text" id="job-title" name="title" value="<?php echo esc_attr( $job['title'] ); ?>" class="regular-text"></td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="job-location"><?php esc_html_e( 'Location', 'converge-applybyvoice' ); ?></label></th>
                                <td><input type="text" id="job-location" name="location" value="<?php echo esc_attr( $job['location'] ?? '' ); ?>" class="regular-text"></td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="job-department"><?php esc_html_e( 'Department', 'converge-applybyvoice' ); ?></label></th>
                                <td><input type="text" id="job-department" name="department" value="<?php echo esc_attr( $job['department'] ?? '' ); ?>" class="regular-text"></td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="job-employment-type"><?php esc_html_e( 'Employment Type', 'converge-applybyvoice' ); ?></label></th>
                                <td>
                                    <?php
                                    $emp_type = $job['employmentType'] ?? '';
                                    $emp_options = array( '' => '—', 'full_time' => 'Full Time', 'part_time' => 'Part Time', 'contract' => 'Contract', 'temporary' => 'Temporary', 'internship' => 'Internship' );
                                    ?>
                                    <select id="job-employment-type" name="employment_type">
                                        <?php foreach ( $emp_options as $val => $label ) : ?>
                                            <option value="<?php echo esc_attr( $val ); ?>" <?php selected( $emp_type, $val ); ?>><?php echo esc_html( $label ); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><label for="job-experience-level"><?php esc_html_e( 'Experience Level', 'converge-applybyvoice' ); ?></label></th>
                                <td>
                                    <?php
                                    $exp_level = $job['experienceLevel'] ?? '';
                                    $exp_options = array( '' => '—', 'entry' => 'Entry Level', 'mid' => 'Mid Level', 'senior' => 'Senior', 'lead' => 'Lead', 'director' => 'Director', 'vp' => 'VP', 'c_level' => 'C-Level' );
                                    ?>
                                    <select id="job-experience-level" name="experience_level">
                                        <?php foreach ( $exp_options as $val => $label ) : ?>
                                            <option value="<?php echo esc_attr( $val ); ?>" <?php selected( $exp_level, $val ); ?>><?php echo esc_html( $label ); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </td>
                            </tr>
                        </table>
                        <p class="submit">
                            <button type="submit" class="button button-primary"><?php esc_html_e( 'Save Basic Info', 'converge-applybyvoice' ); ?></button>
                            <span class="converge-save-status"></span>
                        </p>
                    </form>
                </div>

                <!-- Competencies -->
                <div class="converge-card" id="section-competencies">
                    <h2><?php esc_html_e( 'Competencies', 'converge-applybyvoice' ); ?></h2>
                    <form class="converge-job-form" data-section="competencies" data-job-id="<?php echo esc_attr( $job['id'] ); ?>" id="converge-competencies-form">
                        <input type="hidden" name="competencies" id="converge-competencies-data" value="">

                        <div id="converge-competencies-list">
                            <?php if ( ! empty( $competencies ) ) : ?>
                                <?php foreach ( $competencies as $idx => $comp ) :
                                    $comp_name    = $comp['name'] ?? '';
                                    $comp_weight  = isset( $comp['weight'] ) ? round( $comp['weight'] * 100 ) : 0;
                                    $comp_signals = isset( $comp['signals'] ) ? implode( "\n", $comp['signals'] ) : '';
                                    $comp_questions = isset( $comp['sampleQuestions'] ) ? implode( "\n", $comp['sampleQuestions'] ) : '';
                                ?>
                                    <div class="converge-competency-card" data-index="<?php echo esc_attr( $idx ); ?>">
                                        <div class="converge-competency-header">
                                            <span class="dashicons dashicons-arrow-right-alt2 converge-competency-expand"></span>
                                            <span class="converge-competency-name"><?php echo esc_html( $comp_name ?: __( 'New Competency', 'converge-applybyvoice' ) ); ?></span>
                                            <span class="converge-competency-weight"><?php echo esc_html( $comp_weight . '%' ); ?></span>
                                            <button type="button" class="converge-competency-delete" title="<?php esc_attr_e( 'Delete', 'converge-applybyvoice' ); ?>">&times;</button>
                                        </div>
                                        <div class="converge-competency-body">
                                            <label><?php esc_html_e( 'Name', 'converge-applybyvoice' ); ?></label>
                                            <input type="text" class="comp-name regular-text" value="<?php echo esc_attr( $comp_name ); ?>">

                                            <label><?php esc_html_e( 'Weight (%)', 'converge-applybyvoice' ); ?></label>
                                            <input type="number" class="comp-weight" value="<?php echo esc_attr( $comp_weight ); ?>" min="0" max="100" style="width: 80px;">

                                            <label><?php esc_html_e( 'Signals (one per line)', 'converge-applybyvoice' ); ?></label>
                                            <textarea class="comp-signals" rows="3"><?php echo esc_textarea( $comp_signals ); ?></textarea>

                                            <label><?php esc_html_e( 'Sample Questions (one per line)', 'converge-applybyvoice' ); ?></label>
                                            <textarea class="comp-questions" rows="3"><?php echo esc_textarea( $comp_questions ); ?></textarea>

                                            <div class="converge-competency-actions">
                                                <button type="button" class="button button-small converge-generate-questions">
                                                    <?php esc_html_e( 'Generate Questions', 'converge-applybyvoice' ); ?>
                                                </button>
                                                <span class="converge-save-status converge-gen-status"></span>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>

                        <div style="margin-top: 12px; display: flex; gap: 8px; align-items: center;">
                            <button type="button" id="converge-add-competency" class="button">
                                <?php esc_html_e( '+ Add Competency', 'converge-applybyvoice' ); ?>
                            </button>
                            <?php if ( ! empty( $job['descriptionRaw'] ) ) : ?>
                                <button type="button" id="converge-parse-competencies" class="button"
                                        data-job-id="<?php echo esc_attr( $job['id'] ); ?>"
                                        data-description="<?php echo esc_attr( $job['descriptionRaw'] ); ?>">
                                    <?php esc_html_e( 'Parse from Description', 'converge-applybyvoice' ); ?>
                                </button>
                            <?php endif; ?>
                            <span id="converge-parse-status" class="converge-save-status"></span>
                        </div>

                        <p class="submit">
                            <button type="submit" class="button button-primary"><?php esc_html_e( 'Save Competencies', 'converge-applybyvoice' ); ?></button>
                            <span class="converge-save-status"></span>
                        </p>
                    </form>
                </div>

            <!-- Tab: Conversation -->
            <?php elseif ( $active_tab === 'conversation' ) :
                $bg_config = $conv_config['background'] ?? array();
                $mo_config = $conv_config['motivation'] ?? array();
                $cf_config = $conv_config['culture_fit'] ?? array();
            ?>

                <form class="converge-job-form" data-section="conversation" data-job-id="<?php echo esc_attr( $job['id'] ); ?>">

                    <!-- Background Exploration -->
                    <div class="converge-card converge-config-card">
                        <div class="converge-config-header">
                            <div>
                                <h2><?php esc_html_e( 'Background Exploration', 'converge-applybyvoice' ); ?></h2>
                                <p class="description"><?php esc_html_e( 'Explore the candidate\'s professional background and experience.', 'converge-applybyvoice' ); ?></p>
                            </div>
                            <label class="converge-toggle">
                                <input type="checkbox" name="bg_enabled" value="1" <?php checked( $bg_config['enabled'] ?? true ); ?>>
                                <span class="converge-toggle-slider"></span>
                            </label>
                        </div>

                        <div class="converge-config-body">
                            <div class="converge-ai-field">
                                <label><?php esc_html_e( 'Depth', 'converge-applybyvoice' ); ?></label>
                                <?php $bg_depth = $bg_config['depth'] ?? 2; ?>
                                <div class="converge-range-wrap">
                                    <input type="range" name="bg_depth" min="1" max="5" value="<?php echo esc_attr( $bg_depth ); ?>" class="converge-depth-slider">
                                    <span class="converge-range-value"><?php echo esc_html( $bg_depth ); ?></span>
                                </div>
                                <div class="converge-range-labels">
                                    <span><?php esc_html_e( 'Brief overview', 'converge-applybyvoice' ); ?></span>
                                    <span><?php esc_html_e( 'Deep dive', 'converge-applybyvoice' ); ?></span>
                                </div>
                            </div>

                            <h3><?php esc_html_e( 'Focus Areas', 'converge-applybyvoice' ); ?></h3>
                            <?php
                            $bg_areas = $bg_config['focus_areas'] ?? array();
                            $bg_topics = array(
                                'career_trajectory' => array( __( 'Career Trajectory', 'converge-applybyvoice' ), true ),
                                'recent_role'       => array( __( 'Recent Role Details', 'converge-applybyvoice' ), true ),
                                'key_achievements'  => array( __( 'Key Achievements', 'converge-applybyvoice' ), true ),
                                'role_transitions'  => array( __( 'Role Transitions', 'converge-applybyvoice' ), false ),
                            );
                            foreach ( $bg_topics as $tk => $ti ) :
                                $checked = $bg_areas[ $tk ] ?? $ti[1];
                            ?>
                                <div class="converge-toggle-row">
                                    <div class="converge-toggle-label"><strong><?php echo esc_html( $ti[0] ); ?></strong></div>
                                    <label class="converge-toggle">
                                        <input type="checkbox" name="bg_<?php echo esc_attr( $tk ); ?>" value="1" <?php checked( $checked ); ?>>
                                        <span class="converge-toggle-slider"></span>
                                    </label>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <!-- Motivation Assessment -->
                    <div class="converge-card converge-config-card">
                        <div class="converge-config-header">
                            <div>
                                <h2><?php esc_html_e( 'Motivation Assessment', 'converge-applybyvoice' ); ?></h2>
                                <p class="description"><?php esc_html_e( 'Understand what drives the candidate and their career goals.', 'converge-applybyvoice' ); ?></p>
                            </div>
                            <label class="converge-toggle">
                                <input type="checkbox" name="mo_enabled" value="1" <?php checked( $mo_config['enabled'] ?? true ); ?>>
                                <span class="converge-toggle-slider"></span>
                            </label>
                        </div>

                        <div class="converge-config-body">
                            <h3><?php esc_html_e( 'Topics', 'converge-applybyvoice' ); ?></h3>
                            <?php
                            $mo_questions = $mo_config['questions'] ?? array();
                            $mo_topics = array(
                                'why_this_role'     => array( __( 'Why This Role', 'converge-applybyvoice' ), true ),
                                'why_this_company'  => array( __( 'Why This Company', 'converge-applybyvoice' ), true ),
                                'why_looking_now'   => array( __( 'Why Looking Now', 'converge-applybyvoice' ), true ),
                                'career_goals'      => array( __( 'Career Goals', 'converge-applybyvoice' ), false ),
                                'what_matters_most' => array( __( 'What Matters Most', 'converge-applybyvoice' ), true ),
                            );
                            foreach ( $mo_topics as $tk => $ti ) :
                                $checked = $mo_questions[ $tk ] ?? $ti[1];
                            ?>
                                <div class="converge-toggle-row">
                                    <div class="converge-toggle-label"><strong><?php echo esc_html( $ti[0] ); ?></strong></div>
                                    <label class="converge-toggle">
                                        <input type="checkbox" name="mo_<?php echo esc_attr( $tk ); ?>" value="1" <?php checked( $checked ); ?>>
                                        <span class="converge-toggle-slider"></span>
                                    </label>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <!-- Culture Fit Evaluation -->
                    <div class="converge-card converge-config-card">
                        <div class="converge-config-header">
                            <div>
                                <h2><?php esc_html_e( 'Culture Fit Evaluation', 'converge-applybyvoice' ); ?></h2>
                                <p class="description"><?php esc_html_e( 'Assess alignment with your company culture and values.', 'converge-applybyvoice' ); ?></p>
                            </div>
                            <label class="converge-toggle">
                                <input type="checkbox" name="cf_enabled" value="1" <?php checked( $cf_config['enabled'] ?? false ); ?>>
                                <span class="converge-toggle-slider"></span>
                            </label>
                        </div>

                        <div class="converge-config-body">
                            <h3><?php esc_html_e( 'Areas', 'converge-applybyvoice' ); ?></h3>
                            <?php
                            $cf_areas = $cf_config['areas'] ?? array();
                            $cf_topics = array(
                                'work_environment'    => array( __( 'Work Environment Preference', 'converge-applybyvoice' ), true ),
                                'management_style'    => array( __( 'Management Style Preference', 'converge-applybyvoice' ), true ),
                                'collaboration_style' => array( __( 'Collaboration Style', 'converge-applybyvoice' ), true ),
                                'work_life_balance'   => array( __( 'Work-Life Balance', 'converge-applybyvoice' ), false ),
                                'values_alignment'    => array( __( 'Values Alignment', 'converge-applybyvoice' ), true ),
                            );
                            foreach ( $cf_topics as $tk => $ti ) :
                                $checked = $cf_areas[ $tk ] ?? $ti[1];
                            ?>
                                <div class="converge-toggle-row">
                                    <div class="converge-toggle-label"><strong><?php echo esc_html( $ti[0] ); ?></strong></div>
                                    <label class="converge-toggle">
                                        <input type="checkbox" name="cf_<?php echo esc_attr( $tk ); ?>" value="1" <?php checked( $checked ); ?>>
                                        <span class="converge-toggle-slider"></span>
                                    </label>
                                </div>
                            <?php endforeach; ?>

                            <div class="converge-toggle-row" style="margin-top: 8px;">
                                <div class="converge-toggle-label">
                                    <strong><?php esc_html_e( 'Probe Company Values', 'converge-applybyvoice' ); ?></strong>
                                    <span class="description"><?php esc_html_e( 'Ask about alignment with your company values', 'converge-applybyvoice' ); ?></span>
                                </div>
                                <label class="converge-toggle">
                                    <input type="checkbox" name="cf_probe_values" value="1" <?php checked( $cf_config['probe_company_values'] ?? true ); ?>>
                                    <span class="converge-toggle-slider"></span>
                                </label>
                            </div>
                        </div>
                    </div>

                    <p class="submit">
                        <button type="submit" class="button button-primary"><?php esc_html_e( 'Save Conversation Config', 'converge-applybyvoice' ); ?></button>
                        <span class="converge-save-status"></span>
                    </p>
                </form>

            <!-- Tab: Screening -->
            <?php elseif ( $active_tab === 'screening' ) :
                $reqs = $sr['requirements'] ?? array();
            ?>

                <div class="converge-card" id="section-screening">
                    <h2><?php esc_html_e( 'Screening Data Collection', 'converge-applybyvoice' ); ?></h2>
                    <p class="description"><?php esc_html_e( 'Select what information the AI should collect from candidates during the conversation.', 'converge-applybyvoice' ); ?></p>
                    <form class="converge-job-form" data-section="screening" data-job-id="<?php echo esc_attr( $job['id'] ); ?>">

                        <h3 style="margin-top: 16px;"><?php esc_html_e( 'Information to Collect', 'converge-applybyvoice' ); ?></h3>
                        <?php
                        $collect_fields = array(
                            'collect_salary_expectations' => array( 'Salary Expectations', 'Ask candidates about their compensation expectations' ),
                            'collect_availability'        => array( 'Availability / Start Date', 'When the candidate can start working' ),
                            'collect_location'            => array( 'Current Location', 'Where the candidate is currently based' ),
                            'collect_notice_period'       => array( 'Notice Period', 'How much notice the candidate needs to give' ),
                            'collect_employment_status'   => array( 'Employment Status', 'Whether candidate is currently employed' ),
                            'collect_work_authorization'  => array( 'Work Authorization', 'Whether candidate is authorized to work' ),
                            'collect_relocation'          => array( 'Relocation Willingness', 'Whether candidate is willing to relocate' ),
                        );
                        foreach ( $collect_fields as $field => $info ) :
                            $checked = ! empty( $sr[ $field ] );
                        ?>
                            <div class="converge-toggle-row">
                                <div class="converge-toggle-label">
                                    <strong><?php echo esc_html( $info[0] ); ?></strong>
                                    <span class="description"><?php echo esc_html( $info[1] ); ?></span>
                                </div>
                                <label class="converge-toggle">
                                    <input type="checkbox" name="<?php echo esc_attr( $field ); ?>" value="1" <?php checked( $checked ); ?>>
                                    <span class="converge-toggle-slider"></span>
                                </label>
                            </div>
                        <?php endforeach; ?>

                        <h3 style="margin-top: 24px;"><?php esc_html_e( 'Requirements / Knockout Questions', 'converge-applybyvoice' ); ?></h3>
                        <p class="description"><?php esc_html_e( 'When enabled, the AI will ask about these requirements. Choose whether to flag for review or end the conversation if the requirement is not met.', 'converge-applybyvoice' ); ?></p>

                        <?php
                        $requirement_groups = array(
                            'Legal' => array(
                                'work_authorization'  => 'Work Authorization',
                                'minimum_age'         => 'Minimum Age',
                                'background_check'    => 'Background Check Consent',
                                'drug_screening'      => 'Drug Screening Consent',
                            ),
                            'Availability' => array(
                                'work_weekends'       => 'Available Weekends',
                                'work_evenings'       => 'Available Evenings',
                                'willing_to_travel'   => 'Willing to Travel',
                                'willing_to_relocate' => 'Willing to Relocate',
                            ),
                            'Role' => array(
                                'drivers_license'          => "Driver's License",
                                'reliable_transportation'  => 'Reliable Transportation',
                            ),
                        );
                        foreach ( $requirement_groups as $group_label => $fields ) :
                        ?>
                            <div class="converge-toggle-group-title"><?php echo esc_html( $group_label ); ?></div>
                            <?php foreach ( $fields as $key => $label ) :
                                $req_data = $reqs[ $key ] ?? array();
                                $enabled  = ! empty( $req_data['enabled'] );
                                $behavior = $req_data['behavior'] ?? 'flag';
                            ?>
                                <div class="converge-toggle-row">
                                    <div class="converge-toggle-label">
                                        <strong><?php echo esc_html( $label ); ?></strong>
                                        <div class="converge-requirement-behavior <?php echo $enabled ? 'is-visible' : ''; ?>">
                                            <select name="req_behavior_<?php echo esc_attr( $key ); ?>">
                                                <option value="flag" <?php selected( $behavior, 'flag' ); ?>><?php esc_html_e( 'Flag for review', 'converge-applybyvoice' ); ?></option>
                                                <option value="end" <?php selected( $behavior, 'end' ); ?>><?php esc_html_e( 'End conversation', 'converge-applybyvoice' ); ?></option>
                                            </select>
                                        </div>
                                    </div>
                                    <label class="converge-toggle">
                                        <input type="checkbox" name="req_<?php echo esc_attr( $key ); ?>" value="1" class="converge-requirement-toggle" <?php checked( $enabled ); ?>>
                                        <span class="converge-toggle-slider"></span>
                                    </label>
                                </div>
                            <?php endforeach; ?>
                        <?php endforeach; ?>

                        <!-- Custom Requirements -->
                        <h3 style="margin-top: 24px;"><?php esc_html_e( 'Custom Requirements', 'converge-applybyvoice' ); ?></h3>
                        <p class="description"><?php esc_html_e( 'Add custom knockout questions the AI will ask.', 'converge-applybyvoice' ); ?></p>
                        <?php
                        $custom_reqs = $sr['custom_requirements'] ?? array();
                        ?>
                        <div id="converge-custom-requirements">
                            <?php if ( ! empty( $custom_reqs ) ) :
                                foreach ( $custom_reqs as $idx => $cr ) : ?>
                                    <div class="converge-custom-req-item">
                                        <input type="text" class="custom-req-question regular-text" value="<?php echo esc_attr( $cr['question'] ?? '' ); ?>" placeholder="<?php esc_attr_e( 'e.g., Do you have CPR certification?', 'converge-applybyvoice' ); ?>">
                                        <select class="custom-req-disqualify">
                                            <option value="no" <?php selected( $cr['disqualify_on'] ?? 'no', 'no' ); ?>><?php esc_html_e( 'Disqualify on No', 'converge-applybyvoice' ); ?></option>
                                            <option value="yes" <?php selected( $cr['disqualify_on'] ?? 'no', 'yes' ); ?>><?php esc_html_e( 'Disqualify on Yes', 'converge-applybyvoice' ); ?></option>
                                        </select>
                                        <button type="button" class="converge-custom-req-remove" title="<?php esc_attr_e( 'Remove', 'converge-applybyvoice' ); ?>">&times;</button>
                                    </div>
                                <?php endforeach;
                            endif; ?>
                        </div>
                        <button type="button" id="converge-add-custom-req" class="button button-small" style="margin-top: 8px;">
                            <?php esc_html_e( '+ Add Custom Requirement', 'converge-applybyvoice' ); ?>
                        </button>

                        <p class="submit">
                            <button type="submit" class="button button-primary"><?php esc_html_e( 'Save Screening Settings', 'converge-applybyvoice' ); ?></button>
                            <span class="converge-save-status"></span>
                        </p>
                    </form>
                </div>

            <!-- Tab: Knowledge -->
            <?php elseif ( $active_tab === 'knowledge' ) : ?>

                <div class="converge-card" id="section-knowledge">
                    <h2><?php esc_html_e( 'Job Knowledge Base', 'converge-applybyvoice' ); ?></h2>
                    <p class="description"><?php esc_html_e( 'Information the AI can share with candidates during the conversation.', 'converge-applybyvoice' ); ?></p>
                    <form class="converge-job-form" data-section="knowledge" data-job-id="<?php echo esc_attr( $job['id'] ); ?>">

                        <!-- Compensation -->
                        <div class="converge-kb-card">
                            <h3><?php esc_html_e( 'Compensation', 'converge-applybyvoice' ); ?></h3>
                            <div class="converge-kb-row">
                                <div class="converge-kb-field">
                                    <label for="kb-salary-min"><?php esc_html_e( 'Min Salary', 'converge-applybyvoice' ); ?></label>
                                    <input type="number" id="kb-salary-min" name="salary_range_min" value="<?php echo esc_attr( $kb['salary_range_min'] ?? '' ); ?>" placeholder="50000">
                                </div>
                                <div class="converge-kb-field">
                                    <label for="kb-salary-max"><?php esc_html_e( 'Max Salary', 'converge-applybyvoice' ); ?></label>
                                    <input type="number" id="kb-salary-max" name="salary_range_max" value="<?php echo esc_attr( $kb['salary_range_max'] ?? '' ); ?>" placeholder="70000">
                                </div>
                                <div class="converge-kb-field">
                                    <label for="kb-currency"><?php esc_html_e( 'Currency', 'converge-applybyvoice' ); ?></label>
                                    <select id="kb-currency" name="salary_currency">
                                        <?php
                                        $currencies = array( 'USD' => 'USD', 'EUR' => 'EUR', 'GBP' => 'GBP', 'CAD' => 'CAD', 'AUD' => 'AUD' );
                                        $current_currency = $kb['salary_currency'] ?? 'USD';
                                        foreach ( $currencies as $val => $label ) : ?>
                                            <option value="<?php echo esc_attr( $val ); ?>" <?php selected( $current_currency, $val ); ?>><?php echo esc_html( $label ); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="converge-toggle-row" style="border-bottom: none; padding: 4px 0;">
                                <div class="converge-toggle-label"><strong><?php esc_html_e( 'Negotiable', 'converge-applybyvoice' ); ?></strong></div>
                                <label class="converge-toggle">
                                    <input type="checkbox" name="salary_negotiable" value="1" <?php checked( ! empty( $kb['salary_negotiable'] ) ); ?>>
                                    <span class="converge-toggle-slider"></span>
                                </label>
                            </div>
                            <div class="converge-kb-field">
                                <label for="kb-bonus"><?php esc_html_e( 'Bonus Structure', 'converge-applybyvoice' ); ?></label>
                                <textarea id="kb-bonus" name="bonus_structure" rows="2"><?php echo esc_textarea( $kb['bonus_structure'] ?? '' ); ?></textarea>
                            </div>
                        </div>

                        <!-- Team -->
                        <div class="converge-kb-card">
                            <h3><?php esc_html_e( 'Team', 'converge-applybyvoice' ); ?></h3>
                            <div class="converge-kb-row">
                                <div class="converge-kb-field">
                                    <label for="kb-team-size"><?php esc_html_e( 'Team Size', 'converge-applybyvoice' ); ?></label>
                                    <input type="number" id="kb-team-size" name="team_size" value="<?php echo esc_attr( $kb['team_size'] ?? '' ); ?>" placeholder="e.g. 12">
                                </div>
                                <div class="converge-kb-field">
                                    <label for="kb-direct-reports"><?php esc_html_e( 'Direct Reports', 'converge-applybyvoice' ); ?></label>
                                    <input type="number" id="kb-direct-reports" name="direct_reports" value="<?php echo esc_attr( $kb['direct_reports'] ?? '' ); ?>" placeholder="e.g. 3">
                                </div>
                                <div class="converge-kb-field">
                                    <label for="kb-reporting-to"><?php esc_html_e( 'Reporting To', 'converge-applybyvoice' ); ?></label>
                                    <input type="text" id="kb-reporting-to" name="reporting_to" value="<?php echo esc_attr( $kb['reporting_to'] ?? '' ); ?>" placeholder="e.g. VP of Engineering">
                                </div>
                            </div>
                            <div class="converge-kb-field">
                                <label for="kb-team-desc"><?php esc_html_e( 'Team Description', 'converge-applybyvoice' ); ?></label>
                                <textarea id="kb-team-desc" name="team_description" rows="2"><?php echo esc_textarea( $kb['team_description'] ?? '' ); ?></textarea>
                            </div>
                        </div>

                        <!-- Growth -->
                        <div class="converge-kb-card">
                            <h3><?php esc_html_e( 'Growth Path', 'converge-applybyvoice' ); ?></h3>
                            <div class="converge-kb-row">
                                <div class="converge-kb-field">
                                    <label for="kb-next-role"><?php esc_html_e( 'Next Role', 'converge-applybyvoice' ); ?></label>
                                    <input type="text" id="kb-next-role" name="next_role" value="<?php echo esc_attr( $kb['next_role'] ?? '' ); ?>" placeholder="e.g. Senior Engineer">
                                </div>
                                <div class="converge-kb-field">
                                    <label for="kb-promotion"><?php esc_html_e( 'Promotion Timeline', 'converge-applybyvoice' ); ?></label>
                                    <input type="text" id="kb-promotion" name="promotion_timeline" value="<?php echo esc_attr( $kb['promotion_timeline'] ?? '' ); ?>" placeholder="e.g. 12-18 months">
                                </div>
                            </div>
                            <div class="converge-kb-field">
                                <label for="kb-skills"><?php esc_html_e( 'Skill Development', 'converge-applybyvoice' ); ?></label>
                                <textarea id="kb-skills" name="skill_development" rows="2"><?php echo esc_textarea( $kb['skill_development'] ?? '' ); ?></textarea>
                            </div>
                        </div>

                        <!-- Details -->
                        <div class="converge-kb-card">
                            <h3><?php esc_html_e( 'Details', 'converge-applybyvoice' ); ?></h3>
                            <div class="converge-kb-field">
                                <label for="kb-day-in-life"><?php esc_html_e( 'Day in the Life', 'converge-applybyvoice' ); ?></label>
                                <textarea id="kb-day-in-life" name="day_in_life" rows="3"><?php echo esc_textarea( $kb['day_in_life'] ?? '' ); ?></textarea>
                            </div>
                            <div class="converge-kb-row">
                                <div class="converge-kb-field">
                                    <label for="kb-travel"><?php esc_html_e( 'Travel Requirements', 'converge-applybyvoice' ); ?></label>
                                    <input type="text" id="kb-travel" name="travel_requirements" value="<?php echo esc_attr( $kb['travel_requirements'] ?? '' ); ?>" placeholder="e.g. 20% domestic">
                                </div>
                            </div>
                            <div class="converge-toggle-row" style="border-bottom: none; padding: 4px 0;">
                                <div class="converge-toggle-label"><strong><?php esc_html_e( 'Visa Sponsorship Available', 'converge-applybyvoice' ); ?></strong></div>
                                <label class="converge-toggle">
                                    <input type="checkbox" name="sponsorship_available" value="1" <?php checked( ! empty( $kb['sponsorship_available'] ) ); ?>>
                                    <span class="converge-toggle-slider"></span>
                                </label>
                            </div>
                        </div>

                        <p class="submit">
                            <button type="submit" class="button button-primary"><?php esc_html_e( 'Save Knowledge Base', 'converge-applybyvoice' ); ?></button>
                            <span class="converge-save-status"></span>
                        </p>
                    </form>
                </div>

            <!-- Tab: AI -->
            <?php elseif ( $active_tab === 'ai' ) :
                $ai = $job['aiConversationStyle'] ?? array();
            ?>

                <div class="converge-card" id="section-ai">
                    <h2><?php esc_html_e( 'AI Configuration', 'converge-applybyvoice' ); ?></h2>
                    <p class="description"><?php esc_html_e( 'Configure how the AI interviewer behaves during voice conversations.', 'converge-applybyvoice' ); ?></p>
                    <form class="converge-job-form" data-section="ai" data-job-id="<?php echo esc_attr( $job['id'] ); ?>">

                        <!-- Personality -->
                        <div class="converge-ai-field">
                            <label for="ai-personality"><?php esc_html_e( 'Personality', 'converge-applybyvoice' ); ?></label>
                            <?php
                            $personalities = array(
                                'friendly'     => __( 'Friendly — Warm and approachable', 'converge-applybyvoice' ),
                                'professional' => __( 'Professional — Business-appropriate (default)', 'converge-applybyvoice' ),
                                'casual'       => __( 'Casual — Relaxed and conversational', 'converge-applybyvoice' ),
                                'formal'       => __( 'Formal — Structured and precise', 'converge-applybyvoice' ),
                            );
                            $current_personality = $ai['personality'] ?? 'professional';
                            ?>
                            <select id="ai-personality" name="personality">
                                <?php foreach ( $personalities as $val => $label ) : ?>
                                    <option value="<?php echo esc_attr( $val ); ?>" <?php selected( $current_personality, $val ); ?>><?php echo esc_html( $label ); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Pace -->
                        <div class="converge-ai-field">
                            <label for="ai-pace"><?php esc_html_e( 'Pace', 'converge-applybyvoice' ); ?></label>
                            <?php
                            $paces = array(
                                'relaxed'  => __( 'Relaxed — More time for reflection', 'converge-applybyvoice' ),
                                'moderate' => __( 'Moderate — Balanced flow (default)', 'converge-applybyvoice' ),
                                'brisk'    => __( 'Brisk — Efficient and focused', 'converge-applybyvoice' ),
                            );
                            $current_pace = $ai['pace'] ?? 'moderate';
                            ?>
                            <select id="ai-pace" name="pace">
                                <?php foreach ( $paces as $val => $label ) : ?>
                                    <option value="<?php echo esc_attr( $val ); ?>" <?php selected( $current_pace, $val ); ?>><?php echo esc_html( $label ); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Probing Depth -->
                        <div class="converge-ai-field">
                            <label><?php esc_html_e( 'Probing Depth', 'converge-applybyvoice' ); ?></label>
                            <?php $current_depth = $ai['probing_depth'] ?? 3; ?>
                            <div class="converge-range-wrap">
                                <input type="range" name="probing_depth" min="1" max="5" value="<?php echo esc_attr( $current_depth ); ?>" id="ai-probing-depth">
                                <span class="converge-range-value" id="ai-depth-display"><?php echo esc_html( $current_depth ); ?></span>
                            </div>
                            <div class="converge-range-labels">
                                <span><?php esc_html_e( 'Surface level', 'converge-applybyvoice' ); ?></span>
                                <span><?php esc_html_e( 'Very deep', 'converge-applybyvoice' ); ?></span>
                            </div>
                        </div>

                        <!-- Voice -->
                        <div class="converge-ai-field">
                            <label for="ai-voice"><?php esc_html_e( 'Voice', 'converge-applybyvoice' ); ?></label>
                            <?php
                            $voices_data = array(
                                'Kora'   => array( 'label' => 'Kora', 'gender' => 'Female', 'desc' => 'Warm and engaging. Works well for all roles.', 'id' => 'kora-voice', 'hume_name' => 'KORA', 'recommended' => true ),
                                'Ito'    => array( 'label' => 'Ito', 'gender' => 'Male', 'desc' => 'Clear and professional. Good for technical roles.', 'id' => 'ito-voice', 'hume_name' => 'ITO' ),
                                'Sofia'  => array( 'label' => 'Sofia', 'gender' => 'Female', 'desc' => 'Calm and reassuring. Ideal for thoughtful conversations.', 'id' => 'serene-assistant', 'hume_name' => 'Serene Assistant' ),
                                'James'  => array( 'label' => 'James', 'gender' => 'Male', 'desc' => 'Polished and articulate. Great for professional services.', 'id' => 'male-english-actor', 'hume_name' => 'Male English Actor' ),
                                'Ella'   => array( 'label' => 'Ella', 'gender' => 'Female', 'desc' => 'Warm and expressive. Great for engaging conversations.', 'id' => 'expressive-girl', 'hume_name' => 'Expressive Girl' ),
                                'Marcus' => array( 'label' => 'Marcus', 'gender' => 'Male', 'desc' => 'Confident and professional. Great for corporate roles.', 'id' => '174ddb73-a079-48a7-ab99-8f03b4f9902a', 'hume_name' => 'Marcus' ),
                            );
                            $current_voice = $ai['voice_name'] ?? $ai['voice'] ?? 'Kora';
                            $current_voice_data = $voices_data[ $current_voice ] ?? $voices_data['Kora'];
                            ?>
                            <div class="converge-voice-select-row">
                                <select id="ai-voice" name="voice"
                                        data-voices="<?php echo esc_attr( wp_json_encode( $voices_data ) ); ?>">
                                    <?php foreach ( $voices_data as $key => $v ) : ?>
                                        <option value="<?php echo esc_attr( $key ); ?>"
                                                data-voice-id="<?php echo esc_attr( $v['id'] ); ?>"
                                                data-hume-name="<?php echo esc_attr( $v['hume_name'] ); ?>"
                                                <?php selected( $current_voice, $key ); ?>>
                                            <?php echo esc_html( $v['label'] . ' (' . $v['gender'] . ( ! empty( $v['recommended'] ) ? ', Recommended' : '' ) . ')' ); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <button type="button" class="converge-voice-preview-btn" id="converge-voice-preview"
                                        title="<?php esc_attr_e( 'Preview voice', 'converge-applybyvoice' ); ?>">
                                    <span class="dashicons dashicons-controls-play"></span>
                                </button>
                            </div>
                            <input type="hidden" name="voice_name" id="ai-voice-name" value="<?php echo esc_attr( $current_voice ); ?>">
                            <div class="converge-voice-detail" id="converge-voice-detail">
                                <div>
                                    <strong id="voice-detail-name"><?php echo esc_html( $current_voice_data['label'] ); ?></strong>
                                    <span class="converge-badge converge-badge-gender" id="voice-detail-gender"><?php echo esc_html( $current_voice_data['gender'] ); ?></span>
                                    <?php if ( ! empty( $current_voice_data['recommended'] ) ) : ?>
                                        <span class="converge-badge converge-badge-recommended" id="voice-detail-rec"><?php esc_html_e( 'Recommended', 'converge-applybyvoice' ); ?></span>
                                    <?php else : ?>
                                        <span class="converge-badge converge-badge-recommended" id="voice-detail-rec" style="display:none;"><?php esc_html_e( 'Recommended', 'converge-applybyvoice' ); ?></span>
                                    <?php endif; ?>
                                </div>
                                <div class="converge-voice-desc" id="voice-detail-desc"><?php echo esc_html( $current_voice_data['desc'] ); ?></div>
                                <div class="converge-voice-playing-text" id="voice-playing-text" style="display: none;"><?php esc_html_e( 'Playing...', 'converge-applybyvoice' ); ?></div>
                            </div>
                        </div>

                        <!-- Languages -->
                        <div class="converge-ai-field">
                            <label><?php esc_html_e( 'Allowed Languages', 'converge-applybyvoice' ); ?></label>
                            <p class="description"><?php esc_html_e( 'Candidates can choose from these languages for their conversation.', 'converge-applybyvoice' ); ?></p>
                            <?php
                            $all_languages = array(
                                'en' => 'English', 'es' => 'Spanish', 'fr' => 'French',
                                'de' => 'German', 'it' => 'Italian', 'pt' => 'Portuguese',
                                'ja' => 'Japanese', 'ko' => 'Korean', 'ru' => 'Russian',
                                'hi' => 'Hindi', 'ar' => 'Arabic',
                            );
                            $allowed_langs = $ai['allowed_languages'] ?? array( 'en' );
                            ?>
                            <div class="converge-language-grid">
                                <?php foreach ( $all_languages as $code => $name ) :
                                    $lang_checked = in_array( $code, $allowed_langs, true );
                                ?>
                                    <label>
                                        <input type="checkbox" name="allowed_languages[]" value="<?php echo esc_attr( $code ); ?>" <?php checked( $lang_checked ); ?>>
                                        <?php echo esc_html( $name ); ?>
                                    </label>
                                <?php endforeach; ?>
                            </div>
                            <input type="hidden" name="language" value="<?php echo esc_attr( $ai['language'] ?? 'en' ); ?>">
                        </div>

                        <!-- Prosody -->
                        <div class="converge-ai-field">
                            <div class="converge-toggle-row" style="border-bottom: none; padding: 0;">
                                <div class="converge-toggle-label">
                                    <strong><?php esc_html_e( 'Voice Engagement Analysis', 'converge-applybyvoice' ); ?></strong>
                                    <span class="description"><?php esc_html_e( 'Analyze vocal tone and engagement patterns during conversations. Automatically disabled for EU-based roles.', 'converge-applybyvoice' ); ?></span>
                                </div>
                                <label class="converge-toggle">
                                    <input type="checkbox" name="prosody_enabled" value="1" <?php checked( $ai['prosody_enabled'] ?? true ); ?>>
                                    <span class="converge-toggle-slider"></span>
                                </label>
                            </div>
                        </div>

                        <!-- Custom Instructions -->
                        <div class="converge-ai-field">
                            <label for="ai-instructions"><?php esc_html_e( 'Custom Instructions', 'converge-applybyvoice' ); ?></label>
                            <p class="description"><?php esc_html_e( 'Additional instructions for the AI interviewer. These are included in the system prompt.', 'converge-applybyvoice' ); ?></p>
                            <textarea id="ai-instructions" name="custom_instructions" rows="4" class="large-text"><?php echo esc_textarea( $ai['custom_instructions'] ?? '' ); ?></textarea>
                        </div>

                        <p class="submit">
                            <button type="submit" class="button button-primary"><?php esc_html_e( 'Save AI Configuration', 'converge-applybyvoice' ); ?></button>
                            <span class="converge-save-status"></span>
                        </p>
                    </form>
                </div>

            <?php endif; ?>

        <?php endif; ?>
    <?php endif; ?>
</div>
