<?php
/**
 * Template: Converge Settings (native WP page).
 * Includes connection, sync, widget, AI defaults, billing, team management, and data retention.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

$auth = new Converge_Auth();
$api = new Converge_API( $auth );
$sync = new Converge_Sync( $api );
$connection = $auth->get_connection_info();
$detected_plugin = $sync->detect_plugin();
$last_sync = get_option( 'converge_last_sync', '' );
$last_sync_count = get_option( 'converge_last_sync_count', 0 );
$sync_enabled = get_option( 'converge_sync_enabled', '0' ) === '1';
$sync_frequency = get_option( 'converge_sync_frequency', 'daily' );
$widget_mode = get_option( 'converge_widget_mode', 'float' );

// Load summary for plan info if connected
$summary = null;
if ( $connection['connected'] ) {
    $summary = $api->get( '/api/auth/wp-token', array( 'data' => 'summary' ) );
    if ( is_wp_error( $summary ) ) {
        $summary = null;
    }
}

$plan_type = $summary['planType'] ?? 'starter';
$subscription_status = $summary['subscriptionStatus'] ?? null;
$conversations_this_month = $summary['conversationsThisMonth'] ?? 0;
$monthly_limit = $summary['monthlyLimit'] ?? 5;
?>

<div class="wrap converge-settings">
    <h1><?php esc_html_e( 'Converge Settings', 'converge-applybyvoice' ); ?></h1>

    <!-- Connection Section -->
    <div class="converge-card">
        <h2><?php esc_html_e( 'Account Connection', 'converge-applybyvoice' ); ?></h2>

        <?php if ( $connection['connected'] ) : ?>
            <div class="converge-connection-status converge-connected">
                <span class="dashicons dashicons-yes-alt"></span>
                <div>
                    <strong>
                        <?php
                        printf(
                            esc_html__( 'Connected to %s', 'converge-applybyvoice' ),
                            esc_html( $connection['org_name'] )
                        );
                        ?>
                    </strong>
                    <?php if ( $connection['connected_at'] ) : ?>
                        <br>
                        <span class="description">
                            <?php
                            printf(
                                esc_html__( 'Connected since %s', 'converge-applybyvoice' ),
                                esc_html( date_i18n( get_option( 'date_format' ), $connection['connected_at'] ) )
                            );
                            ?>
                        </span>
                    <?php endif; ?>
                </div>
                <button id="converge-disconnect" class="button button-link-delete">
                    <?php esc_html_e( 'Disconnect', 'converge-applybyvoice' ); ?>
                </button>
            </div>
        <?php else : ?>
            <p><?php esc_html_e( 'Connect your Converge account to sync jobs and manage voice screening from WordPress.', 'converge-applybyvoice' ); ?></p>
            <button id="converge-connect" class="button button-primary button-hero">
                <?php esc_html_e( 'Connect to Converge', 'converge-applybyvoice' ); ?>
            </button>
            <p class="description">
                <?php esc_html_e( 'Don\'t have an account?', 'converge-applybyvoice' ); ?>
                <a href="https://www.convergehiring.com/signup" target="_blank">
                    <?php esc_html_e( 'Sign up free', 'converge-applybyvoice' ); ?>
                </a>
            </p>
        <?php endif; ?>
    </div>

    <?php if ( $connection['connected'] ) : ?>

        <!-- Billing Section (Inline) -->
        <div class="converge-card">
            <h2><?php esc_html_e( 'Billing & Plan', 'converge-applybyvoice' ); ?></h2>

            <div class="converge-billing-info">
                <div class="converge-plan-badge">
                    <span class="converge-plan-name converge-plan-<?php echo esc_attr( $plan_type ); ?>">
                        <?php echo esc_html( ucfirst( $plan_type ) ); ?>
                    </span>
                    <?php if ( $subscription_status === 'active' ) : ?>
                        <span class="converge-subscription-active"><?php esc_html_e( 'Active', 'converge-applybyvoice' ); ?></span>
                    <?php endif; ?>
                </div>

                <?php if ( $monthly_limit > 0 ) : ?>
                    <div class="converge-usage-section">
                        <div class="converge-usage-label">
                            <?php
                            printf(
                                esc_html__( '%d of %d conversations this month', 'converge-applybyvoice' ),
                                intval( $conversations_this_month ),
                                intval( $monthly_limit )
                            );
                            ?>
                        </div>
                        <div class="converge-usage-bar">
                            <?php
                            $usage_pct = $monthly_limit > 0 ? min( 100, round( ( $conversations_this_month / $monthly_limit ) * 100 ) ) : 0;
                            $bar_class = $usage_pct >= 100 ? 'converge-usage-full' : ( $usage_pct >= 80 ? 'converge-usage-warning' : '' );
                            ?>
                            <div class="converge-usage-fill <?php echo esc_attr( $bar_class ); ?>" style="width: <?php echo esc_attr( $usage_pct ); ?>%;"></div>
                        </div>
                    </div>
                <?php endif; ?>

                <div class="converge-billing-actions" style="margin-top: 12px;">
                    <?php if ( in_array( $plan_type, array( 'trial', 'starter' ), true ) ) : ?>
                        <button type="button" class="button button-primary converge-billing-checkout">
                            <?php esc_html_e( 'Upgrade to Pro', 'converge-applybyvoice' ); ?>
                        </button>
                        <span class="description"><?php esc_html_e( '$10 per completed conversation, unlimited roles', 'converge-applybyvoice' ); ?></span>
                    <?php else : ?>
                        <button type="button" class="button converge-billing-portal">
                            <?php esc_html_e( 'Manage Subscription', 'converge-applybyvoice' ); ?>
                        </button>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Job Board Detection -->
        <div class="converge-card">
            <h2><?php esc_html_e( 'Job Board Integration', 'converge-applybyvoice' ); ?></h2>

            <?php if ( $detected_plugin ) : ?>
                <div class="converge-detection-status converge-detected">
                    <span class="dashicons dashicons-admin-plugins"></span>
                    <div>
                        <strong>
                            <?php
                            printf(
                                esc_html__( 'Detected: %s', 'converge-applybyvoice' ),
                                esc_html( $detected_plugin['name'] )
                            );
                            ?>
                        </strong>
                        <br>
                        <span class="description">
                            <?php esc_html_e( 'Jobs from this plugin can be automatically synced to Converge.', 'converge-applybyvoice' ); ?>
                        </span>
                    </div>
                </div>

                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Auto-Sync', 'converge-applybyvoice' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" id="converge-sync-enabled"
                                    <?php checked( $sync_enabled ); ?>
                                    data-nonce="<?php echo esc_attr( wp_create_nonce( 'converge_settings_nonce' ) ); ?>">
                                <?php esc_html_e( 'Automatically sync job listings to Converge', 'converge-applybyvoice' ); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Sync Frequency', 'converge-applybyvoice' ); ?></th>
                        <td>
                            <select id="converge-sync-frequency">
                                <option value="6h" <?php selected( $sync_frequency, '6h' ); ?>><?php esc_html_e( 'Every 6 hours', 'converge-applybyvoice' ); ?></option>
                                <option value="12h" <?php selected( $sync_frequency, '12h' ); ?>><?php esc_html_e( 'Every 12 hours', 'converge-applybyvoice' ); ?></option>
                                <option value="daily" <?php selected( $sync_frequency, 'daily' ); ?>><?php esc_html_e( 'Daily', 'converge-applybyvoice' ); ?></option>
                                <option value="48h" <?php selected( $sync_frequency, '48h' ); ?>><?php esc_html_e( 'Every 48 hours', 'converge-applybyvoice' ); ?></option>
                            </select>
                            <p class="description">
                                <?php esc_html_e( 'New and changed jobs are also synced in real-time when saved.', 'converge-applybyvoice' ); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Last Sync', 'converge-applybyvoice' ); ?></th>
                        <td>
                            <?php if ( $last_sync ) : ?>
                                <?php
                                printf(
                                    esc_html__( '%1$s — %2$d jobs synced', 'converge-applybyvoice' ),
                                    esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $last_sync ) ),
                                    intval( $last_sync_count )
                                );
                                ?>
                            <?php else : ?>
                                <?php esc_html_e( 'Never', 'converge-applybyvoice' ); ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"></th>
                        <td>
                            <button id="converge-sync-now" class="button">
                                <?php esc_html_e( 'Sync Now', 'converge-applybyvoice' ); ?>
                            </button>
                            <span id="converge-sync-status" class="description"></span>
                        </td>
                    </tr>
                </table>
            <?php else : ?>
                <div class="converge-detection-status converge-not-detected">
                    <span class="dashicons dashicons-info-outline"></span>
                    <div>
                        <strong><?php esc_html_e( 'No job board plugin detected', 'converge-applybyvoice' ); ?></strong>
                        <br>
                        <span class="description">
                            <?php esc_html_e( 'Converge supports WP Job Manager, Simple Job Board, and WP Job Openings. Install one of these plugins, or manage jobs directly in the Converge Jobs tab.', 'converge-applybyvoice' ); ?>
                        </span>
                    </div>
                </div>
            <?php endif; ?>
        </div>

        <!-- Widget Display -->
        <div class="converge-card">
            <h2><?php esc_html_e( 'Widget Display', 'converge-applybyvoice' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Choose how the "Apply by Voice" button appears on your job pages.', 'converge-applybyvoice' ); ?></p>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php esc_html_e( 'Button Style', 'converge-applybyvoice' ); ?></th>
                    <td>
                        <select id="converge-widget-mode" data-nonce="<?php echo esc_attr( wp_create_nonce( 'converge_settings_nonce' ) ); ?>">
                            <option value="float" <?php selected( $widget_mode, 'float' ); ?>><?php esc_html_e( 'Floating button (bottom-right corner)', 'converge-applybyvoice' ); ?></option>
                            <option value="button" <?php selected( $widget_mode, 'button' ); ?>><?php esc_html_e( 'Inline button (within page content)', 'converge-applybyvoice' ); ?></option>
                            <option value="auto" <?php selected( $widget_mode, 'auto' ); ?>><?php esc_html_e( 'Auto (only on detected job pages)', 'converge-applybyvoice' ); ?></option>
                        </select>
                    </td>
                </tr>
            </table>
        </div>

        <!-- Team Management -->
        <div class="converge-card">
            <h2><?php esc_html_e( 'Team Members', 'converge-applybyvoice' ); ?></h2>

            <div id="converge-team-list">
                <p class="description"><?php esc_html_e( 'Loading team members...', 'converge-applybyvoice' ); ?></p>
            </div>

            <div class="converge-invite-form" style="margin-top: 16px; padding-top: 16px; border-top: 1px solid #eee;">
                <h3><?php esc_html_e( 'Invite New Member', 'converge-applybyvoice' ); ?></h3>
                <div style="display: flex; gap: 8px; align-items: flex-end;">
                    <div>
                        <label for="converge-invite-email"><?php esc_html_e( 'Email', 'converge-applybyvoice' ); ?></label>
                        <input type="email" id="converge-invite-email" class="regular-text" placeholder="colleague@company.com">
                    </div>
                    <div>
                        <label for="converge-invite-role"><?php esc_html_e( 'Role', 'converge-applybyvoice' ); ?></label>
                        <select id="converge-invite-role">
                            <option value="member"><?php esc_html_e( 'Member', 'converge-applybyvoice' ); ?></option>
                            <option value="admin"><?php esc_html_e( 'Admin', 'converge-applybyvoice' ); ?></option>
                        </select>
                    </div>
                    <button type="button" class="button button-primary converge-invite-btn">
                        <?php esc_html_e( 'Send Invite', 'converge-applybyvoice' ); ?>
                    </button>
                </div>
                <span id="converge-invite-status" class="description" style="margin-top: 8px; display: block;"></span>
            </div>
        </div>

        <!-- AI Defaults -->
        <div class="converge-card">
            <h2><?php esc_html_e( 'AI Defaults', 'converge-applybyvoice' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Organization-level defaults for new jobs. Individual jobs can override these settings.', 'converge-applybyvoice' ); ?></p>
            <div id="converge-ai-defaults">
                <p class="description"><?php esc_html_e( 'Loading AI configuration...', 'converge-applybyvoice' ); ?></p>
            </div>
        </div>

        <!-- Job Automation -->
        <div class="converge-card">
            <h2><?php esc_html_e( 'Job Automation', 'converge-applybyvoice' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Control how jobs are automatically created and published when synced from your job board plugin.', 'converge-applybyvoice' ); ?></p>

            <?php
            $widget_auto_create = $summary['widgetAutoCreate'] ?? false;
            $widget_auto_publish = $summary['widgetAutoPublish'] ?? false;
            ?>

            <div class="converge-toggle-row">
                <div class="converge-toggle-label">
                    <strong><?php esc_html_e( 'Auto-create jobs', 'converge-applybyvoice' ); ?></strong>
                    <span class="description"><?php esc_html_e( 'Automatically create Converge jobs when new job listings are synced.', 'converge-applybyvoice' ); ?></span>
                </div>
                <label class="converge-toggle">
                    <input type="checkbox" id="converge-auto-create" <?php checked( $widget_auto_create ); ?>>
                    <span class="converge-toggle-slider"></span>
                </label>
            </div>
            <div class="converge-toggle-row">
                <div class="converge-toggle-label">
                    <strong><?php esc_html_e( 'Auto-publish jobs', 'converge-applybyvoice' ); ?></strong>
                    <span class="description"><?php esc_html_e( 'Automatically activate newly created jobs so candidates can apply immediately.', 'converge-applybyvoice' ); ?></span>
                </div>
                <label class="converge-toggle">
                    <input type="checkbox" id="converge-auto-publish" <?php checked( $widget_auto_publish ); ?>>
                    <span class="converge-toggle-slider"></span>
                </label>
            </div>
            <div style="margin-top: 12px;">
                <button type="button" class="button button-primary" id="converge-save-automation"><?php esc_html_e( 'Save', 'converge-applybyvoice' ); ?></button>
                <span id="converge-automation-status" class="description" style="margin-left: 8px;"></span>
            </div>
        </div>

        <!-- Candidate Portal -->
        <div class="converge-card">
            <h2><?php esc_html_e( 'Candidate Portal', 'converge-applybyvoice' ); ?></h2>
            <div class="converge-info-banner" style="margin-bottom: 16px;">
                <span class="dashicons dashicons-info-outline" style="color: #3b82f6;"></span>
                <span><?php esc_html_e( 'The candidate portal lets applicants check their application status via a magic link sent to their email after the conversation.', 'converge-applybyvoice' ); ?></span>
            </div>
            <?php
            $portal = $summary['candidatePortalSettings'] ?? array( 'show_application_status' => true, 'show_shortlisted' => true, 'show_archived' => false );
            ?>
            <div class="converge-toggle-row">
                <div class="converge-toggle-label">
                    <strong><?php esc_html_e( 'Show application status', 'converge-applybyvoice' ); ?></strong>
                    <span class="description"><?php esc_html_e( 'Let candidates see their current application status.', 'converge-applybyvoice' ); ?></span>
                </div>
                <label class="converge-toggle">
                    <input type="checkbox" id="converge-portal-status" <?php checked( ! empty( $portal['show_application_status'] ) ); ?>>
                    <span class="converge-toggle-slider"></span>
                </label>
            </div>
            <div class="converge-toggle-row">
                <div class="converge-toggle-label">
                    <strong><?php esc_html_e( 'Show shortlisted status', 'converge-applybyvoice' ); ?></strong>
                    <span class="description"><?php esc_html_e( 'Show candidates when they have been shortlisted.', 'converge-applybyvoice' ); ?></span>
                </div>
                <label class="converge-toggle">
                    <input type="checkbox" id="converge-portal-shortlisted" <?php checked( ! empty( $portal['show_shortlisted'] ) ); ?>>
                    <span class="converge-toggle-slider"></span>
                </label>
            </div>
            <div class="converge-toggle-row">
                <div class="converge-toggle-label">
                    <strong><?php esc_html_e( 'Show archived status', 'converge-applybyvoice' ); ?></strong>
                    <span class="description"><?php esc_html_e( 'Show candidates when their application has been archived.', 'converge-applybyvoice' ); ?></span>
                </div>
                <label class="converge-toggle">
                    <input type="checkbox" id="converge-portal-archived" <?php checked( ! empty( $portal['show_archived'] ) ); ?>>
                    <span class="converge-toggle-slider"></span>
                </label>
            </div>
            <div style="margin-top: 12px;">
                <button type="button" class="button button-primary" id="converge-save-portal"><?php esc_html_e( 'Save', 'converge-applybyvoice' ); ?></button>
                <span id="converge-portal-save-status" class="description" style="margin-left: 8px;"></span>
            </div>
        </div>

        <!-- Data Retention -->
        <div class="converge-card">
            <h2><?php esc_html_e( 'Data Retention', 'converge-applybyvoice' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Control how long conversation data and candidate information is retained.', 'converge-applybyvoice' ); ?></p>
            <?php $retention_days = $summary['dataRetentionDays'] ?? 365; ?>
            <div class="converge-retention-form" style="margin-top: 12px;">
                <div style="display: flex; align-items: center; gap: 12px;">
                    <div>
                        <label for="converge-retention-days"><strong><?php esc_html_e( 'Retention Period (days)', 'converge-applybyvoice' ); ?></strong></label>
                        <input type="number" id="converge-retention-days" class="small-text" min="30" max="3650" value="<?php echo esc_attr( $retention_days ); ?>">
                    </div>
                    <span id="converge-retention-friendly" class="description" style="margin-top: 20px;"></span>
                </div>
                <p class="description" style="margin-top: 8px;">
                    <?php esc_html_e( 'After this period, conversation transcripts, audio recordings, and AI-generated insights will be permanently deleted. Candidate contact information may be retained longer for compliance.', 'converge-applybyvoice' ); ?>
                </p>
                <div style="margin-top: 12px;">
                    <button type="button" class="button button-primary" id="converge-save-retention"><?php esc_html_e( 'Save', 'converge-applybyvoice' ); ?></button>
                    <span id="converge-retention-status" class="description" style="margin-left: 8px;"></span>
                </div>
            </div>
        </div>

    <?php endif; ?>
</div>
